/*
 * Queue - A Queueing system that can be used to handle labs in higher education
 * Copyright (C) 2016-2020  Delft University of Technology
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <https://www.gnu.org/licenses/>.
 */
package nl.tudelft.ewi.queue.controller;

import java.util.List;
import java.util.Map;
import java.util.TreeSet;
import java.util.stream.Collectors;

import nl.tudelft.ewi.queue.dto.view.LabStatisticsViewDto;
import nl.tudelft.ewi.queue.dto.view.RequestFrequencyViewDto;
import nl.tudelft.ewi.queue.dto.view.RequestStatusViewDto;
import nl.tudelft.ewi.queue.model.Assignment;
import nl.tudelft.ewi.queue.model.Lab;
import nl.tudelft.ewi.queue.model.Request;
import nl.tudelft.ewi.queue.model.Room;
import nl.tudelft.ewi.queue.service.LabStatusService;

import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.security.access.prepost.PreAuthorize;
import org.springframework.validation.annotation.Validated;
import org.springframework.web.bind.annotation.GetMapping;
import org.springframework.web.bind.annotation.PathVariable;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.RestController;

@RestController
@Validated
public class CourseStatusController {

	@Autowired
	private LabStatusService lss;

	/**
	 * Creates and returns the data for a histogram regarding the status of requests over time. The data for
	 * requests can be filtered on labs, assignments and rooms to only selectively count.
	 *
	 * @param  id          The id of the course to find status on.
	 * @param  labs        The labs to filter on.
	 * @param  assignments The assignments to filter on.
	 * @param  rooms       The rooms to filter on.
	 * @param  nBuckets    The number of buckets to create. 4 by default (for 4 hours of lab).
	 * @return             A RequestFrequencyViewDto object representing the buckets into which the lab was
	 *                     divided and the number of requests created, open, and handled during each of the
	 *                     created time frames.
	 */
	@GetMapping("/course/{id}/status/freq/request")
	@PreAuthorize("@permissionService.canViewStatus(principal, #id)")
	public RequestFrequencyViewDto getRequestFrequencies(@PathVariable Long id,
			@RequestParam(required = false, defaultValue = "") List<Lab> labs,
			@RequestParam(required = false, defaultValue = "") List<Assignment> assignments,
			@RequestParam(required = false, defaultValue = "") List<Room> rooms,
			@RequestParam(required = false, defaultValue = "5") int nBuckets) {
		List<Request> requests = lss.getFilteredRequests(labs, assignments, rooms);
		TreeSet<Long> buckets = lss.createBucketsOverCourse(labs, nBuckets);

		return new RequestFrequencyViewDto(
				buckets.stream().sorted().collect(Collectors.toList()),
				lss.countCreatedRequestsInBuckets(buckets, requests),
				lss.countOpenRequestsInBuckets(buckets, requests),
				lss.countHandledRequestsInBuckets(buckets, requests));
	}

	/**
	 * Creates and returns the data for a histogram regarding the count of requests statuses. The data for
	 * requests can be filtered on labs, assignments and rooms to only selectively count.
	 *
	 * @param  id          The id of the course to find status on.
	 * @param  labs        The labs to filter on.
	 * @param  assignments The assignments to filter on.
	 * @param  rooms       The rooms to filter on.
	 * @return             The data in the form of a StatusViewDto.
	 */
	@GetMapping("/course/{id}/status/freq/status")
	@PreAuthorize("@permissionService.canViewStatus(principal, #id)")
	public RequestStatusViewDto getRequestStatusFrequencies(@PathVariable Long id,
			@RequestParam(required = false, defaultValue = "") List<Lab> labs,
			@RequestParam(required = false, defaultValue = "") List<Assignment> assignments,
			@RequestParam(required = false, defaultValue = "") List<Room> rooms) {
		List<Request> requests = lss.getFilteredRequests(labs, assignments, rooms);

		return new RequestStatusViewDto(
				lss.countWhere(requests, Request::isPending),
				lss.countWhere(requests, Request::isProcessing),
				lss.countWhere(requests, Request::isApproved),
				lss.countWhere(requests, Request::isRejected),
				lss.countWhere(requests, Request::isNotFound),
				lss.countWhere(requests, Request::isRevoked));
	}

	/**
	 * Gets generic and simple information about the currently filtered labs, assignments and rooms. This
	 * information includes the number of students/assistants participating in the labs, the number of
	 * open/handled requests, the most popular assignment and room, and the average waiting time and
	 * processing time.
	 *
	 * @param  id          The id of the course to find status on.
	 * @param  labs        The labs to filter on.
	 * @param  assignments The assignments to filter on.
	 * @param  rooms       The rooms to filter on.
	 * @return             The data in the form of a LabStatisticsViewDto.
	 */
	@GetMapping("/course/{id}/status/lab/info")
	@PreAuthorize("@permissionService.canViewStatus(principal, #id)")
	public LabStatisticsViewDto getGenericLabInformation(@PathVariable Long id,
			@RequestParam(required = false, defaultValue = "") List<Lab> labs,
			@RequestParam(required = false, defaultValue = "") List<Assignment> assignments,
			@RequestParam(required = false, defaultValue = "") List<Room> rooms) {
		List<Request> requests = lss.getFilteredRequests(labs, assignments, rooms);

		return new LabStatisticsViewDto(
				lss.countDistinctUsers(requests),
				lss.countDistinctAssistants(requests),
				lss.countWhere(requests, Request::isPending),
				lss.countWhere(requests, Request::isHandled),
				lss.mostCountedName(lss.countRequestsPerAssignment(assignments, requests)),
				lss.mostCountedName(lss.countRequestsPerRoom(rooms, requests)),
				lss.averageWaitingTime(requests),
				lss.averageProcessingTime(requests));
	}

	/**
	 * Gets a mapping of the filtered assignments to the number of occurrences of those assignments in the
	 * list of requests filtered by labs and rooms.
	 *
	 * @param  id          The id of the course to find status on.
	 * @param  labs        The labs to filter on.
	 * @param  assignments The assignments to filter on.
	 * @param  rooms       The rooms to filter on.
	 * @return             The names of assignments mapped to the number of times that assignment occurs.
	 */
	@GetMapping("/course/{id}/status/freq/assignment")
	@PreAuthorize("@permissionService.canViewStatus(principal, #id)")
	public Map<String, Long> getAssignmentCounts(@PathVariable Long id,
			@RequestParam(required = false, defaultValue = "") List<Lab> labs,
			@RequestParam(required = false, defaultValue = "") List<Assignment> assignments,
			@RequestParam(required = false, defaultValue = "") List<Room> rooms) {
		List<Request> requests = lss.getFilteredRequests(labs, assignments, rooms);

		return lss.countRequestsPerAssignment(assignments, requests);
	}

	/**
	 * Gets a mapping of the filtered rooms to the number of occurrences of those rooms in the list of
	 * requests filtered by labs and assignments.
	 *
	 * @param  id          The id of the course to find status on.
	 * @param  labs        The labs to filter on.
	 * @param  assignments The assignments to filter on.
	 * @param  rooms       The rooms to filter on.
	 * @return             The names of assignments mapped to the number of times that room occurs.
	 */
	@GetMapping("/course/{id}/status/freq/room")
	@PreAuthorize("@permissionService.canViewStatus(principal, #id)")
	public Map<String, Long> getRoomCounts(@PathVariable Long id,
			@RequestParam(required = false, defaultValue = "") List<Lab> labs,
			@RequestParam(required = false, defaultValue = "") List<Assignment> assignments,
			@RequestParam(required = false, defaultValue = "") List<Room> rooms) {
		List<Request> requests = lss.getFilteredRequests(labs, assignments, rooms);

		return lss.countRequestsPerRoom(rooms, requests);
	}
}
