/*
 * Queue - A Queueing system that can be used to handle labs in higher education
 * Copyright (C) 2016-2020  Delft University of Technology
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <https://www.gnu.org/licenses/>.
 */
package nl.tudelft.ewi.queue.controller;

import javax.servlet.http.HttpServletRequest;

import nl.tudelft.ewi.queue.annotation.AuthenticatedUser;
import nl.tudelft.ewi.queue.model.User;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.core.annotation.AnnotationUtils;
import org.springframework.dao.DataIntegrityViolationException;
import org.springframework.web.bind.annotation.ControllerAdvice;
import org.springframework.web.bind.annotation.ExceptionHandler;
import org.springframework.web.bind.annotation.ResponseStatus;
import org.springframework.web.servlet.ModelAndView;
import org.springframework.web.servlet.mvc.method.annotation.ResponseEntityExceptionHandler;

@ControllerAdvice
public class ErrorControllerAdvice extends ResponseEntityExceptionHandler {

	private static final Logger logger = LoggerFactory.getLogger(ErrorControllerAdvice.class);

	public ErrorControllerAdvice() {
		// TODO Auto-generated constructor stub
	}

	@ExceptionHandler(javax.persistence.EntityNotFoundException.class)
	public static String handle404Exception(Exception e) {
		logger.error("Not found exception: " + e);
		e.printStackTrace();
		return "error/entityNotFound";
	}

	@ExceptionHandler(org.springframework.security.access.AccessDeniedException.class)
	public static String handleDeniedException(@AuthenticatedUser User user, HttpServletRequest request,
			Exception e) {
		logger.error("Access Denied: (User: " + user.getDisplayName() + ", page: " + request.getRequestURI()
				+ "): " + e);
		e.printStackTrace();
		return "error/thouShaltNotPass";
	}

	@ExceptionHandler(DataIntegrityViolationException.class)
	public static String handleSQLException(Exception e) {
		logger.error("SQL Error" + e);
		e.printStackTrace();
		return "error/error";
	}

	@ExceptionHandler(value = Exception.class)
	public static ModelAndView defaultErrorHandler(HttpServletRequest req, Exception e) throws Exception {
		logger.error("A Request (" + req.getRequestURI() + "):  raised " + e);
		e.printStackTrace();
		// If the exception is annotated with @ResponseStatus rethrow it and let
		// the framework handle it - like the OrderNotFoundException example
		// at the start of this post.
		// AnnotationUtils is a Spring Framework utility class.
		if (AnnotationUtils.findAnnotation(e.getClass(), ResponseStatus.class) != null)
			throw e;

		// Otherwise setup and send the user to a default error-view.
		ModelAndView mav = new ModelAndView();
		mav.addObject("exception", e);
		mav.addObject("url", req.getRequestURL());
		mav.setViewName("error/error");
		return mav;
		//return "error/error";
	}
}
