/*
 * Queue - A Queueing system that can be used to handle labs in higher education
 * Copyright (C) 2016-2020  Delft University of Technology
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <https://www.gnu.org/licenses/>.
 */
package nl.tudelft.ewi.queue.controller;

import java.io.IOException;

import nl.tudelft.ewi.queue.csv.EmptyCsvException;
import nl.tudelft.ewi.queue.csv.InvalidCsvException;
import nl.tudelft.ewi.queue.model.Course;
import nl.tudelft.ewi.queue.model.Group;
import nl.tudelft.ewi.queue.repository.CourseRepository;
import nl.tudelft.ewi.queue.repository.GroupRepository;
import nl.tudelft.ewi.queue.service.GroupService;

import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.security.access.prepost.PreAuthorize;
import org.springframework.stereotype.Controller;
import org.springframework.ui.Model;
import org.springframework.web.bind.annotation.*;
import org.springframework.web.multipart.MultipartFile;
import org.springframework.web.servlet.mvc.support.RedirectAttributes;

@Controller
public class GroupController {

	@Autowired
	private GroupRepository groupRepository;

	@Autowired
	private CourseRepository courseRepository;

	@Autowired
	private GroupService groupService;

	@ModelAttribute("page")
	public static String page() {
		return "groups";
	}

	@GetMapping("/course/{id}/groups")
	@PreAuthorize("@permissionService.canManageParticipants(principal, #id)")
	public String view(@PathVariable("id") Long id, Model model) {
		Course course = courseRepository.findById(id).orElseThrow();

		model.addAttribute("course", course);

		return "course/view/groups";
	}

	@PostMapping("/course/{id}/groups/import")
	@PreAuthorize("@permissionService.canManageParticipants(principal, #id)")
	public String importGroups(@PathVariable("id") Long id, @RequestParam("file") MultipartFile csv,
			Model model, RedirectAttributes redirectAttributes)
			throws IOException {
		Course course = courseRepository.findById(id).orElseThrow();
		model.addAttribute("course", course);

		try {
			groupService.importCsv(course, csv);
			redirectAttributes.addFlashAttribute("message", "Groups imported successfully.");
			return "redirect:/course/" + course.getId() + "/groups";

		} catch (EmptyCsvException e) {
			model.addAttribute("csvError", e.getMessage());
		} catch (InvalidCsvException e) {
			model.addAttribute("csvError", e.getMessage());
		}
		return "course/view/groups";
	}

	@GetMapping(value = "/course/{id}/group/{groupId}")
	@PreAuthorize("@permissionService.canManageParticipants(principal, #id)")
	public String view(@PathVariable("id") Long id, @PathVariable("groupId") Long groupId, Model model) {
		Course course = courseRepository.findById(id).orElseThrow();
		Group group = groupRepository.findById(groupId).orElseThrow();
		model.addAttribute("course", course);
		model.addAttribute("group", group);
		return "course/view/group";
	}
}
