/*
 * Queue - A Queueing system that can be used to handle labs in higher education
 * Copyright (C) 2016-2020  Delft University of Technology
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <https://www.gnu.org/licenses/>.
 */
package nl.tudelft.ewi.queue.controller;

import javax.servlet.http.HttpServletRequest;

import nl.tudelft.ewi.queue.annotation.AuthenticatedUser;
import nl.tudelft.ewi.queue.model.Request;
import nl.tudelft.ewi.queue.model.Role;
import nl.tudelft.ewi.queue.model.User;
import nl.tudelft.ewi.queue.repository.RoleRepository;
import nl.tudelft.ewi.queue.service.RequestTableService;

import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.domain.Pageable;
import org.springframework.data.domain.Sort;
import org.springframework.data.querydsl.binding.QuerydslPredicate;
import org.springframework.data.web.PageableDefault;
import org.springframework.security.access.prepost.PreAuthorize;
import org.springframework.stereotype.Controller;
import org.springframework.ui.Model;
import org.springframework.util.MultiValueMap;
import org.springframework.web.bind.annotation.*;
import org.springframework.web.servlet.mvc.support.RedirectAttributes;

import com.querydsl.core.types.Predicate;

@Controller
public class HistoryController {

	@Autowired
	private RoleRepository roleRepository;

	@Autowired
	private RequestTableService requestTableService;

	@ModelAttribute("page")
	public static String page() {
		return "history";
	}

	/**
	 * Maps the student history endpoint to a page.
	 *
	 * This method will fill in the model for the thymeleaf template based on the current session and the
	 * given role and course ids. It fills the model with historic request information through a helper method
	 * in {@link #requestTableService}.
	 *
	 * @param  request  The HttpServletRequest holding the current session.
	 * @param  id       The id of the role to look up.
	 * @param  cId      The id of the course to look up.
	 * @param  model    The model to fill in.
	 * @param  pageable A Pageable object governing pagination of requests.
	 * @return          The path to the corresponding Thymeleaf template.
	 */
	@RequestMapping(value = "/history/course/{cId}/student/{id}", method = { RequestMethod.POST,
			RequestMethod.GET })
	@PreAuthorize("@permissionService.canManageParticipants(principal, #cId)")
	public String listForStudent(HttpServletRequest request,
			RedirectAttributes redirectAttributes,
			@PathVariable("id") Long id,
			@PathVariable("cId") Long cId,
			@RequestParam MultiValueMap<String, String> parameters,
			@QuerydslPredicate(root = Request.class) Predicate predicate,
			Model model, Pageable pageable) {
		if (RequestMethod.POST.name().equals(request.getMethod())) {
			requestTableService.submitFilters(request, parameters, predicate);

			redirectAttributes.addAttribute("page", 0);
			redirectAttributes.addAttribute("size", pageable.getPageSize());

			return "redirect:/history/course/" + cId + "/student/" + id;
		}

		Role role = roleRepository.findById(id).orElseThrow();
		User user = role.getUser();

		requestTableService.fillHistoricTableModel(request, user, pageable, model, cId);

		model.addAttribute("student", user);

		return "history/student";
	}

	/**
	 * Maps the current user history endpoint to a page.
	 *
	 * This method will fill in the model for the thymeleaf template based on the current session and the
	 * authenticated user. It fills the model with historic request information through a helper method in
	 * {@link #requestTableService}.
	 *
	 * @param  request  The HttpServletRequest holding the current session.
	 * @param  user     The currently logged in user to look up.
	 * @param  model    The model to fill in.
	 * @param  pageable A Pageable object governing pagination of requests.
	 * @return          The path to the corresponding Thymeleaf template.
	 */
	@RequestMapping(value = "/history", method = { RequestMethod.POST, RequestMethod.GET })
	public String list(HttpServletRequest request,
			RedirectAttributes redirectAttributes,
			@AuthenticatedUser User user,
			@PageableDefault(sort = "id", direction = Sort.Direction.DESC) Pageable pageable,
			@RequestParam MultiValueMap<String, String> parameters,
			@QuerydslPredicate(root = Request.class) Predicate predicate,
			Model model) {
		if (RequestMethod.POST.name().equals(request.getMethod())) {
			requestTableService.submitFilters(request, parameters, predicate);

			redirectAttributes.addAttribute("page", 0);
			redirectAttributes.addAttribute("size", pageable.getPageSize());

			return "redirect:/history";
		}

		requestTableService.fillHistoricTableModel(request, user, pageable, model, -1L);

		return "history/index";
	}
}
