/*
 * Queue - A Queueing system that can be used to handle labs in higher education
 * Copyright (C) 2016-2020  Delft University of Technology
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <https://www.gnu.org/licenses/>.
 */
package nl.tudelft.ewi.queue.controller;

import java.util.List;
import java.util.stream.Collectors;

import nl.tudelft.ewi.queue.annotation.AuthenticatedUser;
import nl.tudelft.ewi.queue.model.*;
import nl.tudelft.ewi.queue.repository.CourseRepository;
import nl.tudelft.ewi.queue.repository.UserRepository;
import nl.tudelft.ewi.queue.service.CourseService;

import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.support.PagedListHolder;
import org.springframework.data.domain.Pageable;
import org.springframework.security.access.prepost.PreAuthorize;
import org.springframework.security.authentication.AnonymousAuthenticationToken;
import org.springframework.security.core.Authentication;
import org.springframework.security.core.context.SecurityContextHolder;
import org.springframework.stereotype.Controller;
import org.springframework.ui.Model;
import org.springframework.web.bind.annotation.GetMapping;
import org.springframework.web.bind.annotation.ModelAttribute;
import org.springframework.web.bind.annotation.PathVariable;

@Controller
public class HomeController {

	@Autowired
	private UserRepository userRepository;

	@Autowired
	private CourseRepository courseRepository;

	@Autowired
	private CourseService courseService;

	@ModelAttribute("page")
	public static String page() {
		return "main";
	}

	/**
	 * @return The path for the privacy statement.
	 */
	@GetMapping("/privacy")
	public String privacyStatement() {
		return "privacy";
	}

	/**
	 * Maps the index/root url to a page. When the user is logged in correctly with an
	 * AnonymousAuthenticationToken, the user is directed to a dashboard view of the Queue. When the user is
	 * not correctly logged in, they are redirected to the introductory page of the Queue.
	 *
	 * @param  model The model to be filled out when the user has courses and/or labs to show on their
	 *               dashboard.
	 * @return       The path to the corresponding Thymeleaf template.
	 */
	@GetMapping("/")
	public String index(Model model) {
		if (SecurityContextHolder.getContext().getAuthentication() != null
				&& SecurityContextHolder.getContext().getAuthentication().isAuthenticated()
				&& !(SecurityContextHolder.getContext()
						.getAuthentication() instanceof AnonymousAuthenticationToken)) {

			Authentication authentication = SecurityContextHolder.getContext().getAuthentication();
			UserPrincipal userPrincipal = (UserPrincipal) authentication.getPrincipal();

			return dashboard(userRepository.findByUsername(userPrincipal.getUsername()), model);
		}

		return "home/index";
	}

	@GetMapping("/feedback/{id}")
	@PreAuthorize("@permissionService.canViewFeedback(principal, #id)")
	public String feedback(@AuthenticatedUser User user,
			@PathVariable("id") Long id,
			Model model,
			Pageable page) {
		User assistant = userRepository.findById(id).orElseThrow();
		List<Request> filteredList = courseService.requestsWithFeedback(assistant);
		PagedListHolder<Request> requestWithFeedback = new PagedListHolder<>(filteredList);
		requestWithFeedback.setPage(page.getPageNumber());

		model.addAttribute("assistant", assistant);
		model.addAttribute("page", page);
		model.addAttribute("feedback", requestWithFeedback);
		return "home/feedback";
	}

	/**
	 * Maps the user to the dashboard page and inserts the courses and user attributes corresponding to the
	 * given user.
	 *
	 * @param  user  The user to fill the dashboard model for.
	 * @param  model The model to fill.
	 * @return       The path to the dashboard Thymeleaf template.
	 */
	private String dashboard(@AuthenticatedUser User user, Model model) {
		List<Role> availableCourses = user.getRoles().stream()
				.filter(role -> !role.getCourse().isDeleted()
						&& (!role.getCourse().getIsArchived()
								|| role.getUser().isTeacher()))
				.collect(Collectors.toList());

		// TODO this is probably not efficient
		if (user.isAdmin()) {
			List<Course> runningCourses = courseRepository.findAll().stream()
					.filter(course -> !course.getTodaysLabs().isEmpty())
					.collect(Collectors.toList());

			model.addAttribute("runningCourses", runningCourses);
		}

		model.addAttribute("user", user);
		model.addAttribute("availableCourses", availableCourses);

		return "home/dashboard";
	}
}
