/*
 * Queue - A Queueing system that can be used to handle labs in higher education
 * Copyright (C) 2016-2020  Delft University of Technology
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <https://www.gnu.org/licenses/>.
 */
package nl.tudelft.ewi.queue.controller;

import java.time.LocalDateTime;
import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.Optional;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import java.util.stream.Collectors;

import javax.servlet.http.HttpServletRequest;
import javax.validation.Valid;

import nl.tudelft.ewi.queue.annotation.AuthenticatedUser;
import nl.tudelft.ewi.queue.csv.LabCsvHelper;
import nl.tudelft.ewi.queue.model.*;
import nl.tudelft.ewi.queue.repository.*;
import nl.tudelft.ewi.queue.service.JitsiService;
import nl.tudelft.ewi.queue.service.LabService;
import nl.tudelft.ewi.queue.service.RequestService;
import nl.tudelft.ewi.queue.validator.LabValidator;
import nl.tudelft.ewi.queue.viewmodel.RequestCreateModel;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.core.io.ByteArrayResource;
import org.springframework.core.io.Resource;
import org.springframework.http.MediaType;
import org.springframework.http.ResponseEntity;
import org.springframework.security.access.AccessDeniedException;
import org.springframework.security.access.prepost.PreAuthorize;
import org.springframework.stereotype.Controller;
import org.springframework.ui.Model;
import org.springframework.validation.BindingResult;
import org.springframework.validation.annotation.Validated;
import org.springframework.web.bind.WebDataBinder;
import org.springframework.web.bind.annotation.*;
import org.springframework.web.servlet.mvc.support.RedirectAttributes;

import com.fasterxml.jackson.core.JsonProcessingException;
import com.google.common.net.HttpHeaders;

@Controller
@Validated
public class LabController {

	private static final Logger logger = LoggerFactory.getLogger(LabController.class);

	private LabCsvHelper labCsvHelper = new LabCsvHelper();

	@Autowired
	private CourseRepository courseRepository;

	@Autowired
	private LabRepository labRepository;

	@Autowired
	private RoomRepository roomRepository;

	@Autowired
	private LabService labService;

	@Autowired
	private RequestService requestService;

	@Autowired
	private JitsiService jitsiService;

	@Autowired
	private LabValidator labValidator;

	@Autowired
	private RequestTypeRepository requestTypeRepository;

	@Autowired
	private FirstYearMentorGroupRepository firstYearMentorGroupRepository;

	@Autowired
	private UserRepository userRepository;

	@ModelAttribute("page")
	public static String page() {
		return "labs";
	}

	@InitBinder("lab")
	protected void initBinder(WebDataBinder binder) {
		binder.addValidators(labValidator);
	}

	@GetMapping("/lab/{id}")
	@PreAuthorize("@permissionService.canViewLab(principal, #id)")
	public String view(@AuthenticatedUser User user, @PathVariable("id") Long id, Model model) {
		Lab lab = labService.getLab(id);

		List<User> students = lab.getCourse().getStudents().stream().map(Role::getUser)
				.collect(Collectors.toList());
		model.addAttribute("students", students);

		if (lab.isSignOffIntervals()) {
			model.addAttribute("timeslots", labService.getIntervalsForLab(lab));
			model.addAttribute("requests", lab.getPending());
		}

		List<Request> requestsForLab;
		if (user.teaches(lab.getCourse()) || user.manages(lab.getCourse()) || user.assists(lab.getCourse())
				|| user.isAdmin()) {
			requestsForLab = lab.getRequests();
		} else {
			requestsForLab = lab.requestsBy(user);
		}

		// This way, the newest is on top, just like normal request page
		Collections.reverse(requestsForLab);

		model.addAttribute("lab", lab);
		model.addAttribute("requestsForLab", requestsForLab);
		model.addAttribute("course", lab.getCourse());
		if (lab.needsJitsiLink(user)) {
			Request request = lab.getProcessingRequest(user).get();
			model.addAttribute("jitsiURL", jitsiService.getJitsiRoomUrl(request));
		}

		return "lab/view";
	}

	@GetMapping("/lab/{id}/enqueue")
	@PreAuthorize("@permissionService.canEnqueueSelfForLab(principal, #id)")
	public String enqueue(@AuthenticatedUser User user, @PathVariable("id") Long id, Model model) {
		Lab lab = labService.getLab(id);
		if (lab.isEnqueued(user)) {
			return "redirect:/lab/" + id;
		}

		if (lab.isSignOffIntervals()) {
			model.addAttribute("timeslots", labService.getIntervalsForLab(lab));
		}

		model.addAttribute("lab", lab);
		model.addAttribute("course", lab.getCourse());
		model.addAttribute("request", new RequestCreateModel());

		return "lab/enqueue";
	}

	@PostMapping("/lab/{id}/enqueue")
	@PreAuthorize("@permissionService.canEnqueueSelfForLab(principal, #id)")
	public String enqueue(@AuthenticatedUser User user, @PathVariable("id") Long id,
			@Valid RequestCreateModel requestCreateModel, BindingResult bindingResult, Model model) {
		Lab lab = labService.getLab(id);
		if (lab.isEnqueued(user)) {
			return "redirect:/lab/" + id;
		}

		requestCreateModel.setLab(lab);
		if (lab.getCourse().getHasGroups()) {
			Group group = lab.getCourse().getGroup(user);
			requestCreateModel.setRequestEntity(group);
		} else {
			requestCreateModel.setRequestEntity(user);
		}

		// creates a jitsi-room name if this is an online lab
		if (lab.getCommunicationMethod().equals(CommunicationMethod.JITSI_MEET)) {
			requestCreateModel.setJitsi(true);
		}

		if (bindingResult.hasErrors() || requestCreateModel.hasErrors()) {
			model.addAttribute("lab", lab);
			model.addAttribute("course", lab.getCourse());
			model.addAttribute("request", requestCreateModel);
			if (lab.isSignOffIntervals()) {
				model.addAttribute("timeslots", labService.getIntervalsForLab(lab));
			}
			return "lab/enqueue";
		}

		labService.enqueue(requestCreateModel.convert());

		return "redirect:/lab/" + id;
	}

	@PostMapping(value = "/lab/{id}/enqueuestudent", consumes = MediaType.APPLICATION_FORM_URLENCODED_VALUE)
	@PreAuthorize("@permissionService.canEnqueueOthersForLab(principal, #id)")
	public String enqueueOther(@PathVariable("id") Long id,
			@RequestParam("netid") String netid,
			@RequestParam("assignmentid") Long assignmentid,
			@RequestParam("slotStart") String slotStart,
			@RequestParam("slotEnd") String slotEnd) {
		Assignment assignment = labService.getAssignment(assignmentid);
		Lab lab = labService.getLab(id);
		Course course = lab.getCourse();

		User user = userRepository.findByUsername(netid);

		RequestEntity enqueueEntity;
		if (course.getHasGroups()) {
			enqueueEntity = course.getGroup(user);
		} else {
			enqueueEntity = user;
		}

		RequestSlot slot = new RequestSlot(LocalDateTime.parse(slotStart),
				LocalDateTime.parse(slotEnd));

		Request request = new Request(enqueueEntity, assignment, lab.getRooms().get(0),
				requestTypeRepository.findByName("Submission"), "", lab, slot);

		slot.setRequest(request);
		labService.enqueue(request);

		return "redirect:/lab/" + id;
	}

	@PostMapping("/lab/{id}/unenqueue")
	@PreAuthorize("@permissionService.canUnenqueueSelfFromLab(principal, #id)")
	public String unenqueue(@AuthenticatedUser User user, @PathVariable("id") Long id) {
		Lab lab = labService.getLab(id);

		Optional<Request> request = lab.getPendingRequest(user);
		if (lab.getCourse().getHasGroups()) {
			request = lab.getPendingRequest(lab.getCourse().getGroup(user));
		}

		request.ifPresent(request1 -> requestService.revoke(request1));

		return "redirect:/lab/" + id;
	}

	@PostMapping("/lab/{id}/unenqueue/{userId}")
	@PreAuthorize("@permissionService.canUnenqueuOthersFromLab(principal, #id)")
	public String unenqueueOther(@PathVariable("id") Long id,
			@PathVariable("userId") Long userId) {
		Lab lab = labService.getLab(id);

		User user = userRepository.findById(userId).orElseThrow();

		Optional<Request> request = lab.getPendingRequest(user);
		if (lab.getCourse().getHasGroups()) {
			request = lab.getPendingRequest(lab.getCourse().getGroup(user));
		}

		request.ifPresent(request1 -> requestService.revoke(request1));

		return "redirect:/lab/" + id;
	}

	@GetMapping("/course/{id}/lab/create")
	@PreAuthorize("@permissionService.canCreateLab(principal, #id)")
	public String create(@PathVariable("id") Long id, Model model, RedirectAttributes redirectAttributes) {
		Course course = courseRepository.findByIdOrThrow(id);

		if (course.hasNoAssignments()) {
			redirectAttributes.addFlashAttribute("redirectMessage", "Please create an assignment first");
			return "redirect:/course/{id}/assignments";
		}

		if (course.getHasGroups() && course.getGroups().size() == 0) {
			return "redirect:/course/{id}/labs";
		}

		Lab lab = new Lab();
		lab.setCourse(course);
		Iterable<RequestType> requestTypes = requestTypeRepository.findAll();
		Iterable<FirstYearMentorGroup> mentorGroups = firstYearMentorGroupRepository.findAllByActiveIsTrue();

		model.addAttribute("lab", lab);
		model.addAttribute("rooms", roomRepository.findAllByOrderByNameAsc());
		model.addAttribute("course", course);
		model.addAttribute("requesttypes", requestTypes);
		model.addAttribute("mentorgroups", mentorGroups);

		return "lab/create";
	}

	@PostMapping(value = "/course/{id}/lab/create", params = { "addRoom" })
	public String createAddRoom(@PathVariable("id") Long id, @RequestParam("roomName") String name, Lab lab,
			Model model) {
		Course course = courseRepository.findByIdOrThrow(id);

		if (!name.isEmpty()) {
			Room room = new Room(name);
			roomRepository.save(room);
			lab.getRooms().add(room);
		}
		lab.setCourse(course);
		Iterable<RequestType> requestTypes = requestTypeRepository.findAll();

		model.addAttribute("lab", lab);
		model.addAttribute("rooms", roomRepository.findAllByOrderByNameAsc());
		model.addAttribute("course", course);
		model.addAttribute("requesttypes", requestTypes);

		return "lab/create";
	}

	@PostMapping("/course/{id}/lab/create")
	@PreAuthorize("@permissionService.canCreateLab(principal, #id)")
	public String create(@PathVariable("id") Long id,
			@Valid Lab lab,
			BindingResult bindingResult,
			@RequestParam(value = "weekRepeat") Optional<Integer> weekRepeat,
			Model model, RedirectAttributes redirectAttributes) {

		Course course = courseRepository.findByIdOrThrow(id);

		if (bindingResult.hasErrors()) {
			lab.setCourse(course);
			model.addAttribute("lab", lab);
			model.addAttribute("rooms", roomRepository.findAllByOrderByNameAsc());
			model.addAttribute("course", course);
			model.addAttribute("requesttypes", requestTypeRepository.findAll());
			model.addAttribute("mentorgroups", firstYearMentorGroupRepository.findAllByActiveIsTrue());

			return "lab/create";
		}

		labService.saveLab(course, lab, weekRepeat);

		redirectAttributes.addFlashAttribute("message", "Lab created.");

		return "redirect:/lab/" + lab.getId();
	}

	@GetMapping("/lab/{id}/edit")
	@PreAuthorize("@permissionService.canEditLab(principal, #id)")
	public String edit(@PathVariable("id") Long id, Model model) {
		Lab lab = labService.getLab(id);
		List<RequestType> requestTypes = new ArrayList<>();
		requestTypeRepository.findAll().iterator().forEachRemaining(requestTypes::add);

		model.addAttribute("lab", lab);
		model.addAttribute("course", lab.getCourse());
		model.addAttribute("rooms", roomRepository.findAllByOrderByNameAsc());
		model.addAttribute("requesttypes", requestTypes);
		model.addAttribute("mentorgroups", firstYearMentorGroupRepository.findAllByActiveIsTrue());

		return "lab/edit";
	}

	@PostMapping(value = "/lab/{id}", params = { "addRoom" })
	public String addRoom(@PathVariable("id") Long id, @RequestParam("roomName") String name, Lab labData,
			Model model) {
		Lab lab = labService.getLab(id);

		lab.setCommunicationMethod(labData.getCommunicationMethod());
		lab.setSlot(labData.getSlot());
		lab.setRooms(labData.getRooms());
		lab.setAssignments(labData.getAssignments());
		lab.setSlotSelectionOpensAt(labData.getSlotSelectionOpensAt());
		if (!name.isEmpty()) {
			Room room = new Room(name);
			roomRepository.save(room);
			lab.getRooms().add(room);
		}
		Iterable<RequestType> requestTypes = requestTypeRepository.findAll();

		model.addAttribute("lab", lab);
		model.addAttribute("course", lab.getCourse());
		model.addAttribute("rooms", roomRepository.findAllByOrderByNameAsc());
		model.addAttribute("requesttypes", requestTypes);
		model.addAttribute("mentorgroups", firstYearMentorGroupRepository.findAllByActiveIsTrue());

		return "lab/edit";
	}

	@PostMapping("/lab/{id}")
	@PreAuthorize("@permissionService.canEditLab(principal, #id)")
	public String store(@PathVariable("id") Long id, @Valid Lab labData, BindingResult bindingResult,
			Model model, RedirectAttributes redirectAttributes) {
		if (bindingResult.hasErrors()) {
			model.addAttribute("lab", labData);
			model.addAttribute("course", labData.getCourse());
			model.addAttribute("rooms", roomRepository.findAllByOrderByNameAsc());
			model.addAttribute("requesttypes", requestTypeRepository.findAll());
			model.addAttribute("mentorgroups", firstYearMentorGroupRepository.findAllByActiveIsTrue());

			return "lab/edit";
		}

		labService.updateLab(id, labData);

		redirectAttributes.addFlashAttribute("message", "Lab updated.");

		return "redirect:/lab/" + id;
	}

	// --- Remove

	@GetMapping("/lab/{id}/remove")
	@PreAuthorize("@permissionService.canRemoveLab(principal, #id)")
	public String remove(@PathVariable("id") Long id, Model model) {
		Lab lab = labService.getLab(id);

		model.addAttribute("lab", lab);
		model.addAttribute("course", lab.getCourse());

		return "lab/remove";
	}

	@PostMapping("/lab/{id}/remove")
	@PreAuthorize("@permissionService.canRemoveLab(principal, #id)")
	public String destroy(@PathVariable("id") Long id, Model model, RedirectAttributes redirectAttributes) {
		Lab lab = labService.getLab(id);

		labRepository.delete(lab);

		redirectAttributes.addFlashAttribute("message", "Lab removed");

		return "redirect:/course/" + lab.getCourse().getId() + "/labs";
	}

	/**
	 * Takes courseid, assignmentid, and a netid and enqueues the student for a lab in which the assignment
	 * can be signed off.
	 *
	 * @param  assignmentid The assignment to sign off.
	 * @param  netid        The student who wants to sign off.
	 * @return              A redirect to the front page.
	 */
	@PostMapping("/lab/submit")
	// , consumes = MediaType.APPLICATION_FORM_URLENCODED_VALUE
	public String submit(@RequestParam String assignmentid,
			@RequestParam String netid,
			@RequestParam String verification) {

		//todo: add some kind of authentication here :)

		Assignment assignment = labService.getAssignment(Long.parseLong(assignmentid));

		if (!assignment.getVerification().equals(verification)) {
			logger.error("{} tried to enrol with the wrong verification" +
					" code.", netid);
			return "redirect:/";
		}

		Course course = assignment.getCourse();

		User user = userRepository.findByUsername(netid);

		RequestEntity enqueueEntity;
		if (course.getHasGroups()) {
			enqueueEntity = course.getGroup(user);
		} else {
			enqueueEntity = user;
		}

		Optional<Lab> enqueuedLab = labService.randomEnqueue(assignment, enqueueEntity);

		if (enqueuedLab.isPresent()) {
			logger.info("Successful auto enqueue for " + enqueueEntity);
			return "redirect:/lab/" + enqueuedLab.get().getId();
		}

		//todo: some kind of error reply to the caller of this method
		logger.error("No lab exists for this student/assignment combination.");
		return "redirect:/";
	}

	/**
	 * Processes a GET request asking for the CSV file representing the list of requests from the current lab.
	 * This CSV file is created and sent to the requestor through a ResponseEntity.
	 *
	 * Headers are added to the response entity to indicate that the attached resource should be downloaded
	 * and the user should not be redirected to a new page. Additionally, we indicate a name for the file.
	 * https://developer.mozilla.org/en-US/docs/Web/HTTP/Headers/Content-Disposition
	 *
	 * @param  id                      The id of the lab to lookup.
	 * @return                         The ResponseEntity with the Resource to be downloaded.
	 * @throws JsonProcessingException when something goes wrong processing requests.
	 */
	@GetMapping("/lab/{id}/export")
	@PreAuthorize("@permissionService.canExportLab(principal, #id)")
	public ResponseEntity<Resource> downloadLabExport(@PathVariable Long id) throws JsonProcessingException {
		Lab lab = labService.getLab(id);

		return ResponseEntity.ok()
				.header(HttpHeaders.CONTENT_DISPOSITION, "attachment; filename=\"requests-" + id + ".csv\"")
				.body(new ByteArrayResource(labCsvHelper.serializeToCsv(lab.getRequests())));
	}

	@PostMapping("/lab/{id}/close-enqueue/{close}")
	@PreAuthorize("@permissionService.canEditLab(principal, #id)")
	public String closeEnqueue(@PathVariable Long id, @PathVariable boolean close) {
		labService.closeEnqueue(id, close);

		return "redirect:/lab/" + id;
	}

	/**
	 * Handle {@link AccessDeniedException} when this is thrown in the LabController class.
	 *
	 * @param  user    The user for whom this is exception is thrown.
	 * @param  request The http request which was performed.
	 * @param  e       The exception being thrown.
	 * @return         Returns a redirect to the course's enroll page if the user tries to access a lab.
	 *                 Otherwise if the user tries to create/edit a lab, we still give him the error page.
	 */
	@ExceptionHandler(AccessDeniedException.class)
	public String notInLab(@AuthenticatedUser User user, HttpServletRequest request,
			Exception e) {
		Pattern testPattern = Pattern.compile("/lab/([0-9]+)/*");
		Matcher matcher = testPattern.matcher(request.getRequestURI());
		if (matcher.matches() && matcher.groupCount() == 1) {
			Course course = labRepository.findById(Long.parseLong(matcher.group(1))).orElseThrow()
					.getCourse();
			return "redirect:/course/" + course.getId() + "/enroll";
		}

		logger.error("Access Denied: (User: {}, page: {})", user.getDisplayName(), request.getRequestURI());
		logger.error("", e);

		return "error/thouShaltNotPass";
	}
}
