/*
 * Queue - A Queueing system that can be used to handle labs in higher education
 * Copyright (C) 2016-2020  Delft University of Technology
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <https://www.gnu.org/licenses/>.
 */
package nl.tudelft.ewi.queue.controller;

import java.util.List;

import javax.persistence.EntityNotFoundException;

import nl.tudelft.ewi.queue.annotation.AuthenticatedUser;
import nl.tudelft.ewi.queue.model.*;
import nl.tudelft.ewi.queue.repository.CourseRepository;
import nl.tudelft.ewi.queue.repository.NotificationRepository;
import nl.tudelft.ewi.queue.repository.SubscriptionRepository;
import nl.tudelft.ewi.queue.repository.UserRepository;

import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.http.HttpStatus;
import org.springframework.security.access.prepost.PreAuthorize;
import org.springframework.stereotype.Controller;
import org.springframework.ui.Model;
import org.springframework.web.bind.annotation.*;
import org.springframework.web.servlet.mvc.support.RedirectAttributes;

@Controller
public class NotificationController {

	@Autowired
	NotificationRepository notificationRepository;

	@Autowired
	private UserRepository userRepository;

	@Autowired
	CourseRepository courseRepository;

	@Autowired
	private SubscriptionRepository subscriptionRepository;

	@ModelAttribute("page")
	public static String page() {
		return "notifications";
	}

	@RequestMapping(value = "/notifications", method = RequestMethod.GET)
	public static String list(@AuthenticatedUser User user, Model model) {
		model.addAttribute("notifications", user.getNotifications());

		return "notification/index";
	}

	@RequestMapping(value = "/notifications/course/{id}", method = RequestMethod.GET)
	public String filter(@AuthenticatedUser User user, Model model, @PathVariable("id") Long id) {
		model.addAttribute("notifications",
				user.getNotifications().forCourse(courseRepository.findByIdOrThrow(id)));
		model.addAttribute("active", id);

		return "notification/index";
	}

	@RequestMapping(value = "/notification/{id}", method = RequestMethod.GET)
	@PreAuthorize("@permissionService.canViewNotification(principal, #id)")
	public String view(@AuthenticatedUser User user, @PathVariable("id") Long id) {
		Notification notification = getNotification(id);
		notification.setRead(true);

		notificationRepository.save(notification);

		return "redirect:/notifications";
	}

	@RequestMapping(value = "/notifications/clear", method = RequestMethod.GET)
	public String clearAll(@AuthenticatedUser User user, RedirectAttributes redirectAttributes) {
		List<Notification> notificationList = notificationRepository.findByUser(user);
		notificationRepository.deleteAll(notificationList);
		redirectAttributes.addFlashAttribute("message", "Notifications cleared.");

		return "redirect:/notifications";
	}

	protected Notification getNotification(Long id) {
		Notification notification = notificationRepository.findById(id).orElseThrow();

		if (null == notification) {
			throw new EntityNotFoundException("Notification was not found");
		}

		return notification;
	}

	protected Course getCourse(Long id) {
		Course course = courseRepository.findById(id).orElseThrow();

		if (null == course) {
			throw new EntityNotFoundException("Course was not found");
		}

		return course;
	}

	@RequestMapping(value = "/notifications/subscription", method = RequestMethod.POST)
	@ResponseStatus(value = HttpStatus.NO_CONTENT)
	public void subscription(@AuthenticatedUser User user, @RequestBody Subscription subscription) {
		Subscription current = user.getSubscription();
		if (current == null) {
			user.setSubscription(subscription);
			userRepository.save(user);
		} else if (!(current.getEndpoint().equals(subscription.getEndpoint()))) {
			subscriptionRepository.delete(current);
			user.setSubscription(subscription);
			userRepository.save(user);
		}
	}
}
