/*
 * Queue - A Queueing system that can be used to handle labs in higher education
 * Copyright (C) 2016-2020  Delft University of Technology
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <https://www.gnu.org/licenses/>.
 */
package nl.tudelft.ewi.queue.controller;

import static java.util.stream.Collectors.groupingBy;

import java.io.IOException;
import java.util.*;
import java.util.stream.Collectors;
import java.util.stream.Stream;

import javax.persistence.EntityNotFoundException;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import javax.validation.Valid;

import nl.tudelft.ewi.queue.annotation.AuthenticatedUser;
import nl.tudelft.ewi.queue.model.*;
import nl.tudelft.ewi.queue.repository.*;
import nl.tudelft.ewi.queue.service.JitsiService;
import nl.tudelft.ewi.queue.service.RequestService;
import nl.tudelft.ewi.queue.service.RequestTableService;
import nl.tudelft.ewi.queue.views.View;

import org.hibernate.validator.constraints.Range;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.domain.Pageable;
import org.springframework.data.domain.Sort;
import org.springframework.data.querydsl.binding.QuerydslPredicate;
import org.springframework.data.web.PageableDefault;
import org.springframework.data.web.config.EnableSpringDataWebSupport;
import org.springframework.security.access.prepost.PreAuthorize;
import org.springframework.stereotype.Controller;
import org.springframework.ui.Model;
import org.springframework.util.MultiValueMap;
import org.springframework.validation.annotation.Validated;
import org.springframework.web.bind.annotation.*;
import org.springframework.web.servlet.mvc.support.RedirectAttributes;

import com.fasterxml.jackson.annotation.JsonView;
import com.google.common.collect.Lists;
import com.querydsl.core.types.Predicate;

@Controller
@EnableSpringDataWebSupport
@Validated
public class RequestController {

	@Autowired
	private CourseRepository courseRepository;

	@Autowired
	private RequestRepository requestRepository;

	@Autowired
	private UserRepository userRepository;

	@Autowired
	private LabRepository labRepository;

	@Autowired
	private RequestService requestService;

	@Autowired
	private RequestTableService requestTableService;

	@Autowired
	private RequestTypeRepository requestTypeRepository;

	@Autowired
	private RoomRepository roomRepository;

	@Autowired
	private JitsiService jitsiService;

	@ModelAttribute("page")
	public static String page() {
		return "requests";
	}

	// This sorts the latest request first
	private static int compareDates(Request a, Request b) {
		if (a.getCreatedAt().isEqual(b.getCreatedAt()))
			return 0;
		else if (a.getCreatedAt().isAfter(b.getCreatedAt()))
			return -1;
		else
			return 1;
	}

	@RequestMapping(value = "/requests", method = { RequestMethod.POST, RequestMethod.GET })
	@PreAuthorize("@permissionService.canViewRequests(principal)")
	public String list(HttpServletRequest httpRequest,
			@AuthenticatedUser User user, Model model,
			RedirectAttributes redirectAttributes,
			@RequestParam MultiValueMap<String, String> parameters,
			@QuerydslPredicate(root = Request.class) Predicate predicate,
			@PageableDefault(sort = "id", direction = Sort.Direction.DESC, size = 25) Pageable pageable) {
		if (RequestMethod.POST.name().equals(httpRequest.getMethod())) {
			requestTableService.submitFilters(httpRequest, parameters, predicate);

			redirectAttributes.addAttribute("page", 0);
			redirectAttributes.addAttribute("size", pageable.getPageSize());

			return "redirect:/requests";
		}
		requestTableService.fillCurrentTableModel(httpRequest, user, pageable, model);

		return "request/list";
	}

	@RequestMapping(value = "/requests/next/{id}", method = RequestMethod.POST)
	@PreAuthorize("@permissionService.canNextRequest(principal)")
	public String next(HttpServletRequest httpRequest,
			@AuthenticatedUser User user,
			@PathVariable("id") Long id) {
		Predicate predicate = requestTableService.getPredicateFromSession(httpRequest, "/requests");
		Lab lab = labRepository.findById(id).orElseThrow();
		Optional<Request> request = requestService.next(user, lab, predicate);

		return request.map(request1 -> "redirect:/request/" + request1.getId()).orElse("redirect:/requests");
	}

	@RequestMapping(value = "/request/{id}", method = RequestMethod.GET)
	@PreAuthorize("@permissionService.canViewRequest(principal, #id)")
	public String view(@PathVariable("id") Long id, Model model) {
		Request request = getRequest(id);

		model
				.addAttribute("request", request)
				.addAttribute("submissionUrl", requestService.setSubmissionUrl(request))
				.addAttribute("course", request.getLab().getCourse())
				.addAttribute("jitsiURL", jitsiService.getJitsiRoomUrl(request));

		return "request/view";
	}

	@RequestMapping(value = "/request/{id}/approve", method = RequestMethod.GET)
	@PreAuthorize("@permissionService.canFinishRequest(principal, #id)")
	public String approve(@PathVariable("id") Long id, Model model) {
		Request request = getRequest(id);

		model.addAttribute("request", request);

		return "request/approve";
	}

	@RequestMapping(value = "/request/{id}", method = RequestMethod.POST, params = { "approve" })
	@PreAuthorize("@permissionService.canFinishRequest(principal, #id)")
	public String approve(@AuthenticatedUser User user,
			@PathVariable("id") Long id,
			@RequestParam(value = "comment") String comment,
			RedirectAttributes redirectAttributes) {
		Request request = getRequest(id);

		requestService.approve(user, request, comment);

		redirectAttributes.addFlashAttribute("message", "Request approved.");

		return "redirect:/requests";
	}

	@RequestMapping(value = "/request/{id}/reject", method = RequestMethod.GET)
	@PreAuthorize("@permissionService.canFinishRequest(principal, #id)")
	public String reject(@PathVariable("id") Long id, Model model) {
		Request request = getRequest(id);

		model.addAttribute("request", request);

		return "request/reject";
	}

	@RequestMapping(value = "/request/{id}", method = RequestMethod.POST, params = { "reject" })
	@PreAuthorize("@permissionService.canFinishRequest(principal, #id)")
	public String reject(@AuthenticatedUser User user, @PathVariable("id") Long id,
			@RequestParam(value = "comment") String comment,
			@RequestParam(value = "commentForStudent") String commentForStudent,
			RedirectAttributes redirectAttributes) {
		Request request = getRequest(id);

		if (comment.trim().isEmpty() || commentForStudent.trim().isEmpty()) {
			redirectAttributes.addFlashAttribute("message",
					"You need to provide a reason to the student and the TAs.");
			return "redirect:/request/" + id + "/reject";
		}

		requestService.reject(user, request, comment, commentForStudent);

		redirectAttributes.addFlashAttribute("message", "Request rejected.");

		return "redirect:/requests";
	}

	@RequestMapping(value = "/request/{id}/forward", method = RequestMethod.GET)
	@PreAuthorize("@permissionService.canFinishRequest(principal, #id)")
	public String forward(@AuthenticatedUser User user, @PathVariable("id") Long id, Model model) {
		Request request = getRequest(id);
		Course course = request.getLab().getCourse();

		List<User> teachers = course.getTeachers();
		List<User> managers = course.getManagers();
		List<User> assistants = course.getAssistants();

		List<User> others = teachers;
		others.addAll(managers);
		others.addAll(assistants);

		// Do not forward to yourself
		others = others.stream()
				.filter(u -> !u.equals(user))
				.sorted(Comparator.comparing(User::getDisplayName))
				.collect(Collectors.toList());

		model
				.addAttribute("request", request)
				.addAttribute("assistants", others);

		return "request/forward";
	}

	@RequestMapping(value = "/request/{id}/notfound", method = RequestMethod.GET)
	@PreAuthorize("@permissionService.canFinishRequest(principal, #id)")
	public String notFound(@AuthenticatedUser User assistant, @PathVariable("id") Long id, Model model) {
		Request request = getRequest(id);

		requestService.notFound(request, assistant);
		return "redirect:/requests";
	}

	@RequestMapping(value = "/request/{id}", method = RequestMethod.POST, params = { "forward" })
	@PreAuthorize("@permissionService.canFinishRequest(principal, #id)")
	public String forward(@PathVariable("id") Long id,
			@RequestParam(value = "assistant") Long assistantId,
			@RequestParam(value = "comment") String comment,
			RedirectAttributes redirectAttributes) {
		Request request = getRequest(id);

		if (assistantId == -1)
			requestService.forwardToAny(request, comment);
		else
			requestService.forward(request, getAssistant(assistantId), comment);

		redirectAttributes.addFlashAttribute("message", "Request forwarded.");

		return "redirect:/requests";
	}

	@GetMapping(value = "/course/{id}/requests/export", produces = "application/json")
	@PreAuthorize("@permissionService.canEditCourse(principal, #id)")
	@JsonView(View.Summary.class)
	@ResponseBody
	public List<Request> export(@PathVariable("id") Long id) {
		Course course = courseRepository.findByIdOrThrow(id);

		Stream<Request> requests = course.getLabs().stream().flatMap(lab -> lab.getRequests().stream());

		return requests.collect(Collectors.toList());
	}

	@GetMapping(value = "/course/{id}/requests/signofflist.csv")
	@PreAuthorize("@permissionService.canEditCourse(principal, #id)")
	@ResponseBody
	public void completionExport(@PathVariable("id") Long id, HttpServletResponse response)
			throws IOException {
		Course course = courseRepository.findByIdOrThrow(id);

		RequestType submissionRequestType = getSubmissionRequestType();

		Stream<Request> requests = course.getLabs().stream()
				.flatMap(lab -> lab.getRequests().stream())
				.filter(request -> request.getRequestType().equals(submissionRequestType));

		List<Assignment> assignments = course.getAssignments();

		Map<RequestEntity, Map<Assignment, List<Request>>> requestsPerUser = requests.collect(
				groupingBy(Request::getRequestEntity,
						groupingBy(Request::getAssignment)));

		List<List<String>> rows = new ArrayList<>();
		List<String> header = new ArrayList<>();
		header.add("netid");
		header.addAll(assignments.stream().map(Assignment::getName).collect(Collectors.toList()));
		rows.add(header);
		requestsPerUser.forEach((requestEntity, assignmentListMap) -> {
			List<String> row = new ArrayList<>();
			if (requestEntity.isUser()) {
				User student = (User) requestEntity;
				row.add(student.getUsername()); // NetID

				// We loop over the known list of assignments so we're sure to preserve order.
				assignments.forEach(assignment -> {
					// Retrieve the requests for this specific assignment
					List<Request> requestsForAssignment = assignmentListMap.getOrDefault(assignment,
							Lists.newArrayList());
					if (requestsForAssignment.isEmpty()) {
						// Default to NO_REQUEST if there were no requests for this assignment by this user.
						row.add("NO_REQUEST");
					} else {
						// only show result for last entry as this should be the pass/fail
						requestsForAssignment.sort(RequestController::compareDates);
						row.add(String.valueOf(requestsForAssignment.get(0).getStatus()));
					}
				});
			} else {
				// entity is not a user but a group; we need to iterate over the members here; not supported yet
				row.add("Group result not supported yet");
			}
			rows.add(row);
		});
		response.setContentType("text/plain; charset=utf-8");
		response.setHeader("Content-Disposition", "attachment; filename=\"signofflist.csv\"");
		StringBuilder sb = new StringBuilder();
		rows.forEach(row -> sb.append(row.stream().map(Object::toString).collect(Collectors.joining(",")))
				.append("\n"));
		response.getWriter().print(sb.toString());
		response.getWriter().close();
	}

	private RequestType getSubmissionRequestType() {
		RequestType submissionRequestType = null;

		for (RequestType r : requestTypeRepository.findAll()) {
			if (r.getName().equals("Submission")) {
				submissionRequestType = r;
				break;
			}
		}
		return submissionRequestType;
	}

	@RequestMapping(value = "/request/{id}/feedback", method = RequestMethod.POST)
	public String feedback(@AuthenticatedUser User user,
			@PathVariable("id") Long id,
			@RequestParam(value = "feedback", required = false) String feedback,
			@Valid @Range(min = 1L, max = 5L) @RequestParam(value = "rating", required = false) Integer rating,
			RedirectAttributes redirectAttributes) {
		Request request = getRequest(id);
		assert request.getRequestEntity().getId().equals(user.getId());
		if (!request.hasFeedback()) {
			request.setFeedback(feedback);
		}
		if (!request.hasFeedbackRating()) {
			request.setFeedbackRating(rating);
		}
		requestRepository.save(request);
		redirectAttributes.addFlashAttribute("message", "Feedback successfully saved.");
		return "redirect:/request/" + id;
	}

	@RequestMapping(value = "/request/{id}/update-request-info/", method = RequestMethod.POST)
	public String updateRequestInfo(@AuthenticatedUser User user,
			@PathVariable("id") Long id,
			@RequestParam(value = "room") Long roomId,
			@RequestParam(value = "comment", required = false) String comment,
			@RequestParam(value = "question", required = false) String question,
			RedirectAttributes redirectAttributes) {
		Request request = getRequest(id);
		assert request.getRequestEntity().getId().equals(user.getId());
		Room room = roomRepository.findByIdOrThrow(roomId);
		request.setRoom(room);
		if (comment != null) {
			request.setComment(comment);
		} else {
			request.setComment("");
		}
		if (question != null && question.length() >= 15) {
			request.setQuestion(question);
		} else {
			request.setQuestion("");
		}
		requestRepository.save(request);
		redirectAttributes.addFlashAttribute("message", "Your information has been updated.");
		return "redirect:/lab/" + request.getLab().getId();
	}

	private User getAssistant(Long id) {
		User user = userRepository.findById(id).orElseThrow();

		if (null == user) {
			throw new EntityNotFoundException("Entity was not found");
		}

		return user;
	}

	private Request getRequest(Long id) {
		Request request = requestRepository.findById(id).orElseThrow();

		if (null == request) {
			throw new EntityNotFoundException("Entity was not found");
		}

		return request;
	}

}
