/*
 * Queue - A Queueing system that can be used to handle labs in higher education
 * Copyright (C) 2016-2020  Delft University of Technology
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <https://www.gnu.org/licenses/>.
 */
package nl.tudelft.ewi.queue.csv;

import java.util.List;
import java.util.stream.Collectors;

import lombok.AllArgsConstructor;
import lombok.Getter;
import lombok.NoArgsConstructor;
import lombok.Setter;
import nl.tudelft.ewi.queue.model.Group;
import nl.tudelft.ewi.queue.model.Request;
import nl.tudelft.ewi.queue.model.User;

import com.fasterxml.jackson.annotation.JsonProperty;
import com.fasterxml.jackson.annotation.JsonPropertyOrder;
import com.fasterxml.jackson.core.JsonProcessingException;
import com.fasterxml.jackson.dataformat.csv.CsvMapper;
import com.fasterxml.jackson.dataformat.csv.CsvSchema;

/**
 * Helper class for creating and exporting a CSV file for the lab.
 *
 * This class provides a view of a list of requests called RequestRep. The representation provides a NetID to
 * find students by, the start and end time of their requests and the status of the request.
 *
 * If the request is a time slot request, the slot is also included in the representation as start and end
 * times.
 */
public class LabCsvHelper {
	public static final String NET_ID_NAME = "Net ID";
	public static final String CREATED_AT_NAME = "Created at";
	public static final String HANDLED_AT_NAME = "Handled at";
	public static final String SLOT_STARTS_AT_NAME = "Slot starts at";
	public static final String SLOT_ENDS_AT_NAME = "Slot ends at";
	public static final String STATUS_NAME = "Status";
	public static final String TU_DELFT_QUEUE_URL_NAME = "TU Delft Queue URL";

	/**
	 * Representation of the request in an output CSV.
	 *
	 * The representation currently only contains fields representing the request as strings. The
	 * JsonPropertyOrder annotation provides an order to the CSV columns created from this.
	 */
	@Getter
	@Setter
	@NoArgsConstructor
	@AllArgsConstructor
	@JsonPropertyOrder({ NET_ID_NAME, CREATED_AT_NAME, HANDLED_AT_NAME, SLOT_STARTS_AT_NAME,
			SLOT_ENDS_AT_NAME, STATUS_NAME, TU_DELFT_QUEUE_URL_NAME })
	private static final class RequestRepresentation {
		@JsonProperty(NET_ID_NAME)
		private String netId;
		@JsonProperty(CREATED_AT_NAME)
		private String created;
		@JsonProperty(HANDLED_AT_NAME)
		private String handled;
		@JsonProperty(SLOT_STARTS_AT_NAME)
		private String start;
		@JsonProperty(SLOT_ENDS_AT_NAME)
		private String end;
		@JsonProperty(STATUS_NAME)
		private String status;
		@JsonProperty(TU_DELFT_QUEUE_URL_NAME)
		private String url;
	}

	/**
	 * Schema used to define the output CSV file columns.
	 */
	private static final CsvSchema schema = CsvSchema.builder()
			.addColumn(NET_ID_NAME)
			.addColumn(CREATED_AT_NAME)
			.addColumn(HANDLED_AT_NAME)
			.addColumn(SLOT_STARTS_AT_NAME)
			.addColumn(SLOT_ENDS_AT_NAME)
			.addColumn(STATUS_NAME)
			.addColumn(TU_DELFT_QUEUE_URL_NAME)
			.build().withHeader();

	/**
	 * Serializes the given list of requests into a list of request representations, which are then serialized
	 * into a CSV file in the form of a byte array.
	 *
	 * @param  requests                The list of requests to serialize.
	 * @return                         The serialized CSV file as a byte array.
	 * @throws JsonProcessingException when something goes wrong processing the request representations.
	 */
	public byte[] serializeToCsv(List<Request> requests) throws JsonProcessingException {
		final var writer = new CsvMapper()
				.writer().with(schema);

		final var reps = requests.stream().map(r -> {
			var rr = new RequestRepresentation(null,
					r.getCreatedAt().toString(),
					(r.getHandledAt() == null) ? null : r.getHandledAt().toString(),
					(r.getSlot() == null) ? null : r.getSlot().getOpensAt().toString(),
					(r.getSlot() == null) ? null : r.getSlot().getClosesAt().toString(),
					r.getStatus().displayName(),
					"queue.tudelft.nl/request/" + r.getId());

			if (r.getRequestEntity() instanceof User) {
				rr.setNetId(((User) r.getRequestEntity()).getStudentNumber() + "");
			} else {
				List<User> users = ((Group) r.getRequestEntity()).getMembers();
				rr.setNetId(users.stream()
						.map(u -> u.getStudentNumber() + "")
						.collect(Collectors.joining(" ")));
			}

			return rr;
		}).collect(Collectors.toList());

		return writer
				.writeValueAsBytes(reps);
	}
}
