/*
 * Queue - A Queueing system that can be used to handle labs in higher education
 * Copyright (C) 2016-2020  Delft University of Technology
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <https://www.gnu.org/licenses/>.
 */
package nl.tudelft.ewi.queue.csv;

import java.io.*;
import java.util.List;

import nl.tudelft.ewi.queue.model.Assistant;
import nl.tudelft.ewi.queue.model.Course;
import nl.tudelft.ewi.queue.model.DefaultRole;
import nl.tudelft.ewi.queue.model.Manager;
import nl.tudelft.ewi.queue.model.Role;
import nl.tudelft.ewi.queue.model.Teacher;
import nl.tudelft.ewi.queue.model.User;

import org.springframework.web.multipart.MultipartFile;

import com.fasterxml.jackson.annotation.JsonCreator;
import com.fasterxml.jackson.annotation.JsonProperty;
import com.fasterxml.jackson.databind.RuntimeJsonMappingException;
import com.fasterxml.jackson.dataformat.csv.CsvSchema;

public class UserCsvHelper {
	private static final CsvSchema schema = CsvSchema.builder()
			.setSkipFirstDataRow(true)
			.addColumn("netId")
			.addColumn("role")
			.setColumnSeparator(';')
			.build();

	private String netId;

	private DefaultRole role;

	public String getNetId() {
		return netId;
	}

	public void setNetId(String netId) {
		this.netId = netId;
	}

	public void setRole(DefaultRole role) {
		this.role = role;
	}

	@JsonCreator
	public UserCsvHelper(@JsonProperty("netId") String netId, @JsonProperty("role") String role)
			throws RuntimeJsonMappingException, DefaultRole.UnknownRoleException {
		this.netId = netId;
		this.role = DefaultRole.valueOfExt(role);
	}

	public Role getRole(User staff, Course course) throws IllegalStateException {
		Role role = null;
		switch (this.role) {
			case ROLE_ASSISTANT:
				role = new Assistant(staff, course);
				break;
			case ROLE_TEACHER:
				role = new Teacher(staff, course);
				break;
			case ROLE_MANAGER:
				role = new Manager(staff, course);
				break;
			default:
				throw new IllegalStateException("Unexpected value: " + this.role);
		}
		return role;
	}

	public DefaultRole getRole() {
		return role;
	}

	/***
	 * Reads a csv file and parses it using jackson and returns a list of objects containing (netId, role).
	 *
	 * @param  csvFile
	 * @return                     A list of objects (netId, role).
	 * @throws IOException
	 * @throws EmptyCsvException   it the csv file is empty.
	 * @throws InvalidCsvException if the csv file is invalid (eg: unknown role, too many delimiters).
	 */
	public static List<UserCsvHelper> readCsv(MultipartFile csvFile)
			throws IOException, EmptyCsvException, InvalidCsvException {
		return CsvHelper.readCsv(csvFile, UserCsvHelper.class, schema);
	}

}
