/*
 * Queue - A Queueing system that can be used to handle labs in higher education
 * Copyright (C) 2016-2020  Delft University of Technology
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <https://www.gnu.org/licenses/>.
 */
package nl.tudelft.ewi.queue.dialect;

import java.util.Set;

import nl.tudelft.ewi.queue.model.UserPrincipal;
import nl.tudelft.ewi.queue.repository.UserRepository;

import org.springframework.security.core.Authentication;
import org.springframework.security.core.context.SecurityContextHolder;
import org.thymeleaf.context.IExpressionContext;
import org.thymeleaf.expression.IExpressionObjectFactory;

import com.google.common.collect.Sets;

public class AuthenticatedExpressionObjectFactory implements IExpressionObjectFactory {

	private UserRepository userRepository;

	private static final String AUTHENTICATED_OBJECT_NAME = "authenticated";

	AuthenticatedExpressionObjectFactory(UserRepository userRepository) {
		this.userRepository = userRepository;
	}

	@Override
	public Set<String> getAllExpressionObjectNames() {
		return Sets.newHashSet(AUTHENTICATED_OBJECT_NAME);
	}

	@Override
	public Object buildObject(IExpressionContext context, String expressionObjectName) {
		if (AUTHENTICATED_OBJECT_NAME.equals(expressionObjectName)) {
			return getUser();
		}
		return null;
	}

	@Override
	public boolean isCacheable(String expressionObjectName) {
		return false;
	}

	protected Object getUser() {
		Authentication authentication = SecurityContextHolder.getContext().getAuthentication();

		if (authentication == null) {
			return null;
		}

		if (authentication.getPrincipal() instanceof UserPrincipal) {
			UserPrincipal userPrincipal = (UserPrincipal) authentication.getPrincipal();

			return userRepository.findById(userPrincipal.getUser().getId()).orElseThrow();
		}

		return authentication.getPrincipal();
	}
}
