/*
 * Queue - A Queueing system that can be used to handle labs in higher education
 * Copyright (C) 2016-2020  Delft University of Technology
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <https://www.gnu.org/licenses/>.
 */
package nl.tudelft.ewi.queue.dialect;

import java.util.List;
import java.util.Set;

import nl.tudelft.ewi.queue.model.Request;

import org.springframework.util.MultiValueMap;
import org.springframework.web.util.UriComponentsBuilder;
import org.thymeleaf.context.IExpressionContext;
import org.thymeleaf.expression.IExpressionObjectFactory;

import com.google.common.collect.Sets;

public class FilterExpressionObjectFactory implements IExpressionObjectFactory {

	private static final String FILTER_OBJECT_NAME = "filter";

	@Override
	public Set<String> getAllExpressionObjectNames() {
		return Sets.newHashSet(FILTER_OBJECT_NAME);
	}

	@Override
	public Object buildObject(IExpressionContext context, String expressionObjectName) {
		if (FILTER_OBJECT_NAME.equals(expressionObjectName)) {
			return new Filter();
		}
		return null;
	}

	@Override
	public boolean isCacheable(String expressionObjectName) {
		return false;
	}

	public class Filter {
		/**
		 * Builds a URI corresponding that includes the given state in its query parameters.
		 *
		 * @param  state The current state of the filter as a Map.
		 * @return       The relative URI built from the state.
		 */
		public String buildUrlMultiValueMap(MultiValueMap<String, String> state) {
			return buildUrlMultiValueMap(state, "");
		}

		/**
		 * Builds a URI corresponding that includes the given state in its query parameters and uses the given
		 * path.
		 *
		 * @param  state The current state of the filter as a Map.
		 * @param  path  The path for the URI to have.
		 * @return       The relative URI built from the state.
		 */
		public String buildUrlMultiValueMap(MultiValueMap<String, String> state, String path) {
			return UriComponentsBuilder.fromPath(path)
					.queryParam("size", state.getFirst("size"))
					.queryParam("page", state.getFirst("page"))
					.build(false).encode().toUriString();
		}

		/**
		 * Takes a MultiValueMap representing the filter state and updates it.
		 *
		 * @param  state       The state of the filter as a map.
		 * @param  updateField The field that must be updated.
		 * @param  updateValue The value with which to update the field.
		 * @return             The new (relative) URL built from the state.
		 */
		public String updateUrlMultiValueMap(MultiValueMap<String, String> state,
				String updateField, String updateValue) {
			return updateUrlMultiValueMap(state, updateField, updateValue, "");
		}

		/**
		 * Takes a MultivalueMap representing the filter state and updates it.
		 *
		 * @param  state       The state of the filter as a map.
		 * @param  updateField The field that must be updated.
		 * @param  updateValue The value with which to update the field.
		 * @param  path        The path of the output URL.
		 * @return             The new URL built from the MultivalueMap.
		 */
		public String updateUrlMultiValueMap(MultiValueMap<String, String> state,
				String updateField, String updateValue,
				String path) {
			state.set(updateField, updateValue);
			return buildUrlMultiValueMap(state, path);
		}

		/**
		 * Checks whether the given key and value pair is an entry in the given state.
		 *
		 * @param  key   The key to check.
		 * @param  value The value to check.
		 * @param  state The state of the filter as a map.
		 * @return       {@code true} when the given key-value pair is in the given state, {@code false}
		 *               otherwise.
		 */
		public boolean valueInMultiMap(String key, String value, MultiValueMap<String, String> state) {
			List<String> valueSet = state.get(key);
			if (valueSet != null) {
				return valueSet.contains(value);
			}
			return false;
		}

		/**
		 * Maps the status of a request to a fitting bootstrap background color.
		 *
		 * @param  status The status of the request.
		 * @return        The selected bootstrap background color.
		 */
		public String mapStatusToBootstrapColor(int status) {
			Request.Status values = Request.Status.values()[status];
			String bgColor = "";
			switch (values) {
				case PICKED:
				case PENDING:
				case ASSIGNED:
					bgColor = "bg-idle";
					break;
				case CANCELLED:
				case REVOKED:
				case PROCESSING:
					bgColor = "bg-processing";
					break;
				case APPROVED:
					bgColor = "bg-approved";
					break;
				case REJECTED:
					bgColor = "bg-rejected";
					break;
				case NOTFOUND:
					bgColor = "bg-notfound";
					break;
				case FORWARDED:
					bgColor = "bg-forwarded";
					break;
			}
			return bgColor;
		}
	}

}
