/*
 * Queue - A Queueing system that can be used to handle labs in higher education
 * Copyright (C) 2016-2020  Delft University of Technology
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <https://www.gnu.org/licenses/>.
 */
package nl.tudelft.ewi.queue.dialect;

import java.time.LocalDateTime;
import java.time.temporal.ChronoUnit;
import java.util.Set;

import org.thymeleaf.context.IExpressionContext;
import org.thymeleaf.expression.IExpressionObjectFactory;

import com.google.common.collect.Sets;

public class PrettyTimeExpressionObjectFactory implements IExpressionObjectFactory {

	private static final String PRETTY_TIME_OBJECT_NAME = "prettyTime";

	@Override
	public Set<String> getAllExpressionObjectNames() {
		return Sets.newHashSet(PRETTY_TIME_OBJECT_NAME);
	}

	@Override
	public Object buildObject(IExpressionContext context, String expressionObjectName) {
		if (PRETTY_TIME_OBJECT_NAME.equals(expressionObjectName)) {
			return new PrettyTime();
		}
		return null;
	}

	@Override
	public boolean isCacheable(String expressionObjectName) {
		return false;
	}

	private class PrettyTime {
		public String diffForHumans(LocalDateTime time) {
			if (null == time) {
				return "";
			}

			return diffForHumans(time, LocalDateTime.now());
		}

		public String diffForHumans(LocalDateTime time, LocalDateTime other) {
			long years = ChronoUnit.YEARS.between(time, other);
			long months = ChronoUnit.MONTHS.between(time, other);
			long days = ChronoUnit.DAYS.between(time, other);
			long hours = ChronoUnit.HOURS.between(time, other);
			long minutes = ChronoUnit.MINUTES.between(time, other);
			long seconds = ChronoUnit.SECONDS.between(time, other);

			if (years > 0) {
				return display(ChronoUnit.YEARS, years);
			} else if (months > 0) {
				return display(ChronoUnit.MONTHS, months);
			} else if (days > 0) {
				return display(ChronoUnit.DAYS, days);
			} else if (hours > 0) {
				return display(ChronoUnit.HOURS, hours);
			} else if (minutes > 0) {
				return display(ChronoUnit.MINUTES, minutes);
			} else {
				return display(ChronoUnit.SECONDS, seconds);
			}
		}

		protected String display(ChronoUnit unit, long count) {
			String suffix;

			if (count > 1) {
				suffix = "s";
			} else {
				suffix = "";
			}

			switch (unit) {
				case YEARS:
					return count + " year" + suffix + " ago";
				case MONTHS:
					return count + " month" + suffix + " ago";
				case DAYS:
					return count + " day" + suffix + " ago";
				case HOURS:
					return count + " hour" + suffix + " ago";
				case MINUTES:
					return count + " minute" + suffix + " ago";
				default:
					return count + " second" + suffix + " ago";
			}
		}
	}
}
