/*
 * Queue - A Queueing system that can be used to handle labs in higher education
 * Copyright (C) 2016-2020  Delft University of Technology
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <https://www.gnu.org/licenses/>.
 */
package nl.tudelft.ewi.queue.dto.patch;

import org.springframework.validation.Errors;

/**
 * Interface for patch DTOs. A Patch DTO contains fields to represent the data in a patch request and
 * implements the apply and hasErrors methods.
 *
 * The apply method is used to apply the patch changes to the target entity object.
 *
 * @param <D> The type of the target entity.
 */
public interface Patch<D> {
	/**
	 * Applies the changes of the patch to a target entity object.
	 *
	 * @param data The object to change with this patch.
	 */
	void apply(D data);

	/**
	 * Checks whether the patch is valid and adds to the Errors object if it is not valid.
	 *
	 * @param errors The errors object additional errors get added to.
	 */
	void validateImpl(Errors errors);

	/**
	 * Checks whether the patch is valid and adds to the Errors object if it is not valid.
	 *
	 * @param  errors The errors object additional errors get added to.
	 * @return        {@code true} when the patch is deemed valid.
	 */
	default boolean validate(Errors errors) {
		validateImpl(errors);
		return errors.hasErrors();
	}

	/**
	 * Checks whether the given string is non-null. If it non-null, it is not allowed to be empty. If s is
	 * empty, this method adds an error to the given Errors object.
	 *
	 * @param field  The field
	 * @param s      The name of the
	 * @param errors The accumulated validation errors.
	 */
	default void nonEmpty(String field, String s, Errors errors) {
		if (s != null && s.isBlank()) {
			errors.rejectValue(field, "Field must be either null or non-empty.");
		}
	}
}
