/*
 * Queue - A Queueing system that can be used to handle labs in higher education
 * Copyright (C) 2016-2020  Delft University of Technology
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <https://www.gnu.org/licenses/>.
 */
package nl.tudelft.ewi.queue.helper;

import java.time.Duration;
import java.time.LocalDateTime;
import java.util.Properties;

import javax.mail.Message;
import javax.mail.MessagingException;
import javax.mail.Session;
import javax.mail.Transport;
import javax.mail.internet.InternetAddress;
import javax.mail.internet.MimeMessage;

import nl.tudelft.ewi.queue.model.Assignment;
import nl.tudelft.ewi.queue.model.Group;
import nl.tudelft.ewi.queue.model.RequestSlot;
import nl.tudelft.ewi.queue.model.User;

public class EmailHelper {

	private static EmailHelper ourInstance = new EmailHelper();
	private String from;
	private Properties properties;
	private Session session;

	private EmailHelper() {
		from = "noreply@tudelft.nl";

		properties = new Properties();
		properties.put("host", "smtp.tudelft.nl");

		session = Session.getDefaultInstance(properties);
	}

	/**
	 * Get the email helper singleton.
	 *
	 * @return The instance of EmailHelper.
	 */
	public static EmailHelper getInstance() {
		return ourInstance;
	}

	//todo: could consider having one notify method: notify entity of timeslot
	/**
	 * Notifies the student of their new slot for a specific assignment. The email will be send from
	 * noreply@tudelft.nl.
	 *
	 * @param user       The user which needs to be notified of their new slot.
	 * @param slot       The slot to which the user is assigned.
	 * @param assignment The assignment the user will be signing off.
	 */
	public void notifyStudentOfTimeSlot(User user, RequestSlot slot,
			Assignment assignment) {
		try {
			MimeMessage messageobj = new MimeMessage(this.session);

			messageobj.setFrom(new InternetAddress(from));
			messageobj.addRecipient(Message.RecipientType.TO,
					new InternetAddress(user.getEmail()));

			messageobj.setSubject("New enrollment " +
					"in the queue");

			Duration slotDuration = Duration.between(slot.getOpensAt(), slot.getClosesAt());

			LocalDateTime time = slot.getOpensAt();
			messageobj.setText("Dear" + user.getDisplayName() + "\n " +
					"You have been " + "enqueued for assignment "
					+ assignment.getName() + ".\n" +
					"Your slot starts at " + time.toString() + "and " +
					"will take roughly "
					+ slotDuration.toMinutes() + " minutes. \n \n" +
					"Kind regards, \n The queue team.");

			Transport.send(messageobj);
		} catch (MessagingException e) {
			e.printStackTrace();
		}
	}

	/**
	 * Notifies all members of a group of the timeslot they are enqueued in.
	 *
	 * @param group       The group whose members must be notified.
	 * @param requestSlot The timeslot the members are enqueued for.
	 * @param assignment  The assignment the group wants to sign of.
	 */
	public void notifyGroupOfTimeSlot(Group group, RequestSlot requestSlot, Assignment assignment) {
		for (User user : group.getMembers()) {
			notifyStudentOfTimeSlot(user, requestSlot, assignment);
		}
	}
}
