/*
 * Queue - A Queueing system that can be used to handle labs in higher education
 * Copyright (C) 2016-2020  Delft University of Technology
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <https://www.gnu.org/licenses/>.
 */
package nl.tudelft.ewi.queue.model;

import java.io.Serializable;
import java.time.LocalDateTime;
import java.util.ArrayList;
import java.util.List;
import java.util.Random;

import javax.persistence.*;

import lombok.Getter;
import lombok.Setter;
import nl.tudelft.ewi.queue.views.View;

import org.bouncycastle.util.encoders.Hex;
import org.hibernate.annotations.GenericGenerator;
import org.hibernate.annotations.SQLDelete;

import com.fasterxml.jackson.annotation.JsonView;

@Entity
@SQLDelete(sql = "UPDATE assignment SET deleted_at = NOW() WHERE id = ?")
@Getter
@Setter
public class Assignment implements Serializable, Comparable<Assignment> {

	private static final long serialVersionUID = -4104521574160856982L;

	@Id
	@GeneratedValue(strategy = GenerationType.IDENTITY)
	private Long id;

	@ManyToOne
	private Course course;

	@ManyToMany
	private List<Lab> labs = new ArrayList<>();

	@OneToMany(mappedBy = "assignment")
	private List<Request> requests;

	@JsonView(View.Summary.class)
	private String name;

	@GenericGenerator(name = "uuid-gen", strategy = "uuid")
	@GeneratedValue(generator = "uuid-gen")
	private String verification;

	@ManyToMany
	private List<User> assistants = new ArrayList<>();

	@SuppressWarnings("unused")
	private LocalDateTime deletedAt;

	public Assignment() {
		this.verification = randomVerificationCode();
	}

	public Assignment(Course course, String name) {
		this.course = course;
		this.name = name;
		this.verification = randomVerificationCode();

	}

	public Long getId() {
		return id;
	}

	public Course getCourse() {
		return course;
	}

	public void setCourse(Course course) {
		this.course = course;
	}

	public String getName() {
		return name;
	}

	public void setName(String name) {
		this.name = name;
	}

	public List<Lab> getLabs() {
		return labs;
	}

	public void setLabs(List<Lab> labs) {
		this.labs = labs;
	}

	public void addLab(Lab lab) {
		labs.add(lab);

		if (!lab.getAssignments().contains(this)) {
			lab.addAssignment(this);
		}
	}

	public String getVerification() {
		return this.verification;
	}

	private String randomVerificationCode() {
		byte[] resBuf = new byte[50];
		new Random().nextBytes(resBuf);
		return new String(Hex.encode(resBuf));
	}

	public boolean hasTA(User assistant) {
		return this.assistants.contains(assistant);
	}

	/**
	 * Check to see if a student is allowed ot handle a specific assignment. If there are no TA's speified for
	 * an assignment, then everybody is allowed to handle it.
	 *
	 * @param  assistant Assistant wanting to handle the assignment.
	 * @return           Whether the list is empty or the assistant is in the list.
	 */
	public boolean allowedToHandleAssignment(User assistant) {
		return this.assistants.isEmpty() || this.assistants.contains(assistant);
	}

	@Override
	public String toString() {
		return name;
	}

	@Override
	public int compareTo(Assignment assignment) {
		return name.compareTo(assignment.getName());
	}
}
