/*
 * Queue - A Queueing system that can be used to handle labs in higher education
 * Copyright (C) 2016-2020  Delft University of Technology
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <https://www.gnu.org/licenses/>.
 */
package nl.tudelft.ewi.queue.model;

import java.io.Serializable;
import java.text.DecimalFormat;
import java.time.LocalDateTime;
import java.util.*;
import java.util.function.Predicate;
import java.util.stream.Collectors;

import javax.persistence.*;
import javax.validation.constraints.NotBlank;
import javax.validation.constraints.NotNull;

import org.hibernate.annotations.SQLDelete;
import org.hibernate.annotations.Where;
import org.springframework.cache.annotation.Cacheable;

import com.fasterxml.jackson.core.JsonProcessingException;

@Entity
@SQLDelete(sql = "UPDATE course SET deleted_at = NOW() WHERE id = ?")
@Where(clause = "deleted_at IS NULL")
public class Course implements Serializable {

	private static final long serialVersionUID = 254709728043436131L;

	@Id
	@GeneratedValue(strategy = GenerationType.IDENTITY)
	private Long id;

	@NotBlank
	private String name;

	@NotBlank
	private String code;

	@OneToMany(mappedBy = "course")
	private List<Lab> labs = new ArrayList<>();

	@OneToMany(mappedBy = "course", cascade = { CascadeType.ALL }, orphanRemoval = true)
	@Where(clause = "deleted_at IS NULL")
	private List<Assignment> assignments = new ArrayList<>();

	@OneToMany(mappedBy = "course", cascade = { CascadeType.MERGE }, orphanRemoval = true)
	private List<Role> roles = new ArrayList<>();

	private String submissionUrl;

	@NotNull
	private Boolean hasGroups = false;

	@OneToMany(mappedBy = "course", cascade = { CascadeType.MERGE }, orphanRemoval = true)
	private List<Group> groups = new ArrayList<>();

	@NotNull
	private Boolean isArchived = false;

	@SuppressWarnings("unused")
	private LocalDateTime deletedAt;

	public Course() {
	}

	public Course(String name, String code) {
		this.name = name;
		this.code = code;
	}

	public Course(String name, String code, String submissionUrl) {
		this(name, code);
		this.submissionUrl = submissionUrl;
	}

	public Course(String name, String code, String submissionUrl, Boolean hasGroups) {
		this(name, code, submissionUrl);
		this.hasGroups = hasGroups;
	}

	public Course(String name, String code, String submissionUrl, Boolean hasGroups, Boolean isArchived) {
		this(name, code, submissionUrl, hasGroups);
		this.isArchived = isArchived;
	}

	public Long getId() {
		return id;
	}

	public void setId(Long id) {
		this.id = id;
	}

	public String getName() {
		return name;
	}

	public void setName(String name) {
		this.name = name;
	}

	public String getCode() {
		return code;
	}

	public void setCode(String code) {
		this.code = code;
	}

	public List<Lab> getLabs() {
		return labs;
	}

	public List<Lab> getTodaysLabs() {
		return labs.stream()
				.filter(l -> l.getSlot().isTodayish() || l.slotSelectionIsOpen())
				.collect(Collectors.toList());
	}

	public List<Lab> getOldLabs() {
		final LocalDateTime now = LocalDateTime.now();
		return labs.stream()
				.filter(l -> l.getSlot().getClosesAt().isBefore(now))
				.collect(Collectors.toList());
	}

	public List<Lab> getActiveLabs() {
		final LocalDateTime now = LocalDateTime.now();
		return labs.stream()
				.filter(l -> l.getSlot().contains(now))
				.collect(Collectors.toList());
	}

	public void setLabs(List<Lab> labs) {
		this.labs = labs;
	}

	public void addLab(Lab lab) {
		labs.add(lab);

		if (!this.equals(lab.getCourse())) {
			lab.setCourse(this);
		}
	}

	public List<Assignment> getAssignments() {
		Collections.sort(assignments);
		return assignments;
	}

	public void setAssignments(List<Assignment> assignments) {
		this.assignments = assignments;
	}

	public void addAssignment(Assignment assignment) {
		assignments.add(assignment);

		if (!this.equals(assignment.getCourse())) {
			assignment.setCourse(this);
		}
	}

	public List<Role> getRoles() {
		return roles;
	}

	public void setRoles(List<Role> roles) {
		this.roles = roles;
	}

	public void addRole(Role role) {
		roles.add(role);

		if (!this.equals(role.getCourse())) {
			role.setCourse(this);
		}
	}

	public List<Role> getStudents() {
		return roles.stream()
				.filter(r -> r instanceof Student)
				.collect(Collectors.toList());
	}

	private List<User> getUsers(Predicate<Role> f) {
		return roles.stream()
				.filter(f)
				.map(Role::getUser)
				.collect(Collectors.toList());
	}

	public List<User> getAssistants() {
		return getUsers(r -> r instanceof Assistant);
	}

	public List<User> getManagers() {
		return getUsers(r -> r instanceof Manager);
	}

	public List<User> getTeachers() {
		return getUsers(r -> r instanceof Teacher);
	}

	public List<User> getPrivileged() {
		return getUsers(r -> r instanceof Assistant || r instanceof Manager || r instanceof Teacher);
	}

	public List<Role> getRoles(Class<Role> role) {
		return roles.stream()
				.filter(role::isInstance)
				.collect(Collectors.toList());
	}

	public boolean isEnrolled(User user) {
		return roles.stream()
				.anyMatch(r -> user.equals(r.getUser()));
	}

	public boolean isGroupOfThisCourse(Group group) {
		return groups.stream()
				.anyMatch(r -> group.getId().equals(r.getId()));
	}

	public Group getGroup(User user) {
		Optional<Group> group = groups.stream().filter(g -> g.hasMember(user)).findFirst();
		return group.orElse(null);
	}

	public String getSubmissionUrl() {
		return submissionUrl;
	}

	public void setSubmissionUrl(String submissionURL) {
		this.submissionUrl = submissionURL;
	}

	public Boolean getHasGroups() {
		return hasGroups;
	}

	public void setHasGroups(Boolean hasGroups) {
		this.hasGroups = hasGroups;
	}

	public List<Group> getGroups() {
		return groups;
	}

	public void setGroups(List<Group> groups) {
		this.groups = groups;
	}

	public void addGroup(Group group) {
		//todo: should there be a check for the hasGroups boolean here?
		if (!isGroupOfThisCourse(group)) {
			groups.add(group);
		}
	}

	public void removeGroup(Group toRemove) {
		groups.removeIf(group -> group.getId().equals(toRemove.getId()));
	}

	private OptionalDouble averageWaiting() {
		double totalWaitingTime = 0;
		int labCount = 0;
		for (Lab lab : this.getTodaysLabs()) {
			OptionalDouble waiting = lab.averageWaitingDouble();
			if (waiting.isPresent()) {
				totalWaitingTime += waiting.getAsDouble();
				labCount += 1;
			}
		}
		return calculateAverage(totalWaitingTime, labCount);
	}

	private OptionalDouble averageProcessing() {
		double totalWaitingTime = 0;
		int labCount = 0;
		for (Lab lab : this.getTodaysLabs()) {
			OptionalDouble waiting = lab.averageProcessingDouble();
			if (waiting.isPresent()) {
				totalWaitingTime += waiting.getAsDouble();
				labCount += 1;
			}
		}
		return calculateAverage(totalWaitingTime, labCount);
	}

	public Optional<String> averageWaitingFormatted() {
		return formattedDouble(this.averageWaiting());
	}

	public Optional<String> averageProcessingFormatted() {
		return formattedDouble(this.averageProcessing());
	}

	private Optional<String> formattedDouble(OptionalDouble optionalDouble) {
		DecimalFormat df = new DecimalFormat("#.##");

		if (!optionalDouble.isPresent()) {
			return Optional.empty();
		}
		return Optional.of(df.format(optionalDouble.getAsDouble()));
	}

	private OptionalDouble calculateAverage(double total, double amount) {
		OptionalDouble avgWaitingTime = OptionalDouble.empty();
		if (amount != 0) {
			avgWaitingTime = OptionalDouble.of(total / amount);
		}
		return avgWaitingTime;
	}

	public Long activeAssistants() {
		ArrayList<User> activeAssistants = new ArrayList<>();
		for (Lab lab : this.getTodaysLabs()) {
			activeAssistants.addAll(lab.activeAssistants().collect(Collectors.toList()));
		}
		return activeAssistants.stream().distinct().count();
	}

	public Long studentsInTheQueue() {
		Long studentsInQueue = 0L;
		for (Lab lab : this.getTodaysLabs()) {
			studentsInQueue += lab.nrOfStudentsInQueue();
		}
		return studentsInQueue;
	}

	@Cacheable("totalPerHour")
	public String requestsTotalPerHour() throws JsonProcessingException {
		ArrayList<Request> requests = new ArrayList<>();
		for (Lab lab : this.getTodaysLabs()) {
			requests.addAll(lab.getRequests());
		}
		return Request.requestsPerHour(earliestLabOpensAt(), latestClosedLab(), requests);
	}

	@Cacheable("courseRejectedPerHour")
	public String requestsRejectedPerHour() throws JsonProcessingException {
		ArrayList<Request> requestsRejected = new ArrayList<>();
		for (Lab lab : this.getTodaysLabs()) {
			requestsRejected.addAll(lab.getRejected());
		}
		return Request.requestsPerHour(earliestLabOpensAt(), latestClosedLab(), requestsRejected);
	}

	@Cacheable("courseApprovedPerHour")
	public String requestsApprovedPerHour() throws JsonProcessingException {
		ArrayList<Request> requestsApproved = new ArrayList<>();
		for (Lab lab : this.getTodaysLabs()) {
			requestsApproved.addAll(lab.getApproved());
		}
		return Request.requestsPerHour(earliestLabOpensAt(), latestClosedLab(), requestsApproved);
	}

	@Cacheable("courseHandledPerHour")
	public String requestsHandledPerHour() throws JsonProcessingException {
		ArrayList<Request> requestsHandled = new ArrayList<>();
		for (Lab lab : this.getTodaysLabs()) {
			requestsHandled.addAll(lab.getHandled());
		}
		return Request.requestsPerHour(earliestLabOpensAt(), latestClosedLab(), requestsHandled);
	}

	private LocalDateTime earliestLabOpensAt() {
		LocalDateTime earliestOpen = LocalDateTime.now();
		for (Lab lab : this.getTodaysLabs()) {
			if (earliestOpen.isAfter(lab.getSlot().getOpensAt()))
				earliestOpen = lab.getSlot().getOpensAt();
		}
		return earliestOpen;
	}

	private LocalDateTime latestClosedLab() {
		LocalDateTime latestClosed = LocalDateTime.now();
		for (Lab lab : this.getTodaysLabs()) {
			if (latestClosed.isBefore(lab.getSlot().getClosesAt()))
				latestClosed = lab.getSlot().getClosesAt();
		}
		return latestClosed;
	}

	public boolean hasNoAssignments() {
		return this.assignments.isEmpty();
	}

	public Boolean getIsArchived() {
		return isArchived;
	}

	public void setIsArchived(Boolean archived) {
		isArchived = archived;
	}

	/**
	 * @return Whether this course is deleted.
	 */
	public Boolean isDeleted() {
		return deletedAt != null;
	}

	public List<Room> getUsedRooms() {
		return labs.stream()
				.flatMap(lab -> lab.getFilteredRooms().stream())
				.distinct()
				.sorted()
				.collect(Collectors.toList());
	}

	@Override
	public String toString() {
		return name + " (" + code + ")";
	}
}
