/*
 * Queue - A Queueing system that can be used to handle labs in higher education
 * Copyright (C) 2016-2020  Delft University of Technology
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <https://www.gnu.org/licenses/>.
 */
package nl.tudelft.ewi.queue.model;

import static java.time.LocalDateTime.now;

import java.io.Serializable;
import java.text.DecimalFormat;
import java.time.LocalDateTime;
import java.time.format.DateTimeFormatter;
import java.util.ArrayList;
import java.util.Comparator;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.OptionalDouble;
import java.util.stream.Collectors;
import java.util.stream.Stream;

import javax.annotation.Nullable;
import javax.persistence.*;
import javax.validation.Valid;
import javax.validation.constraints.Max;
import javax.validation.constraints.Min;
import javax.validation.constraints.NotEmpty;
import javax.validation.constraints.NotNull;

import lombok.Getter;
import lombok.Setter;
import nl.tudelft.ewi.queue.views.View;

import org.hibernate.annotations.SQLDelete;
import org.hibernate.annotations.Where;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.cache.annotation.Cacheable;
import org.springframework.cache.annotation.EnableCaching;
import org.springframework.format.annotation.DateTimeFormat;

import com.fasterxml.jackson.annotation.JsonView;
import com.fasterxml.jackson.core.JsonProcessingException;

@Entity
@SQLDelete(sql = "UPDATE lab SET deleted_at = NOW(), slot_id = NULL WHERE id = ?")
@Where(clause = "deleted_at IS NULL")
@EnableCaching
@Getter
@Setter
public class Lab implements Serializable {
	/**
	 *
	 */
	private static final long serialVersionUID = 4861827801551639673L;

	private static final Logger logger = LoggerFactory.getLogger(Lab.class);

	public final static String TIME_FORMAT = "dd/MM/yyyy HH:mm";

	@Id
	@GeneratedValue(strategy = GenerationType.IDENTITY)
	@JsonView(View.Summary.class)
	private Long id;

	private String title;
	private boolean isFeedbackLab;

	@Valid
	@ManyToOne
	private Course course = new Course();

	@ManyToMany
	@Where(clause = "deleted_at IS NULL")
	@NotEmpty
	@Valid
	private List<Assignment> assignments = new ArrayList<>();

	@OneToOne(cascade = { CascadeType.ALL }, orphanRemoval = true)
	@Valid
	@JsonView(View.Summary.class)
	private LabSlot slot;

	@Enumerated(EnumType.STRING)
	@NotNull
	private CommunicationMethod communicationMethod;

	//@ManyToMany(cascade = {CascadeType.PERSIST, CascadeType.MERGE})
	@ManyToMany
	@NotEmpty
	@Valid
	private List<Room> rooms = new ArrayList<>();

	@OneToMany(mappedBy = "lab", cascade = { CascadeType.ALL })
	@Valid
	private List<Request> requests = new ArrayList<>();

	@ManyToMany
	private List<FirstYearMentorGroup> allowedMentorGroups = new ArrayList<>();

	private boolean allowWithoutMentorGroup = false;

	@ManyToMany
	@Valid
	private List<RequestType> allowedRequestTypes = new ArrayList<>();

	private boolean signOffIntervals;

	// only relevant if sign off intervals are used
	// sets the time it takes to sign off one assignment
	@Min(value = 1L, message = "Interval time must be positive")
	private Long intervalTime;

	// only relevant if sign off intervals are used
	// sets the number of groups that can select a single time slot
	@Min(value = 1L, message = "Capacity must be positive")
	private Long capacity;

	@DateTimeFormat(pattern = TIME_FORMAT)
	private LocalDateTime slotSelectionOpensAt;

	private boolean examLab = false;

	@Min(value = 10L, message = "Percentage must be at least 10")
	@Max(value = 100L, message = "Percentage cannot exceed 100")
	@Nullable
	private Integer examLabPercentage;

	private boolean enqueueClosed = false;

	@SuppressWarnings("unused")
	private LocalDateTime deletedAt;

	public Lab() {
	}

	public Lab(Course course, LabSlot slot, List<Room> rooms,
			boolean feedbackLab) {
		this.course = course;
		this.slot = slot;
		this.rooms = rooms;
		this.isFeedbackLab = feedbackLab;
	}

	//    public Lab(Course course, LabSlot slot, List<Room> rooms, boolean isFeedbackLab,
	//               List<Assignment> assignments) {
	//        this.course = course;
	//    }

	public Integer getExamLabPercentage() {
		return null == examLabPercentage ? 0 : examLabPercentage;
	}

	public String getTitle() {
		if (title != null) {
			return title;
		}
		return "";
	}

	/**
	 * Get the queue of requests that are pending or assigned, ascescending by creation.
	 *
	 * @return
	 */
	public List<Request> getQueue() {
		return getRequests().stream()
				.filter(Request::isQueued)
				.sorted(Comparator.comparing(Request::getCreatedAt))
				.collect(Collectors.toList());
	}

	/**
	 * Get the queue of pending requests sorted descending by creation.
	 *
	 * @return
	 */
	public List<Request> getPending() {
		return getRequests().stream()
				.filter(Request::isPending)
				.sorted(Comparator.comparing(Request::getCreatedAt))
				.collect(Collectors.toList());
	}

	/**
	 * Get processing requests sorted descending by creation.
	 *
	 * @return
	 */
	public List<Request> getProcessing() {
		return getRequests().stream()
				.filter(Request::isProcessing)
				.sorted(Comparator.comparing(Request::getCreatedAt))
				.collect(Collectors.toList());
	}

	/**
	 * Get archived requests sorted descending by creation.
	 *
	 * @return
	 */
	public List<Request> getArchived() {
		return getRequests().stream()
				.filter(Request::isArchived)
				.sorted(Comparator.comparing(Request::getCreatedAt))
				.collect(Collectors.toList());
	}

	/**
	 * Get handled requests sorted descending by creation.
	 *
	 * @return
	 */
	public List<Request> getHandled() {
		return getRequests().stream()
				.filter(Request::isHandled)
				.sorted(Comparator.comparing(Request::getCreatedAt))
				.collect(Collectors.toList());
	}

	public List<Request> getApproved() {
		return getRequests().stream()
				.filter(Request::isApproved)
				.sorted(Comparator.comparing(Request::getCreatedAt))
				.collect(Collectors.toList());
	}

	public List<Request> getRejected() {
		return getRequests().stream()
				.filter(Request::isRejected)
				.sorted(Comparator.comparing(Request::getCreatedAt))
				.collect(Collectors.toList());
	}

	/**
	 * Returns an Optional with the pending request for the given user if it exists, or an empty Optional
	 * otherwise.
	 *
	 * @param  requestEntity
	 *
	 * @return
	 */
	public Optional<Request> getPendingRequest(RequestEntity requestEntity) {
		return getQueue().stream()
				.filter(r -> r.getRequestEntity().equals(requestEntity))
				.findFirst();
	}

	/**
	 * Returns an Optional with the processing request for the given user if it exists, or an empty Optional
	 * otherwise.
	 *
	 * @param  requestEntity the entity for whose request is being searched for
	 * @return
	 */
	public Optional<Request> getProcessingRequest(RequestEntity requestEntity) {
		Optional<Request> req = getRequests().stream()
				.filter(r -> r.isProcessing() && r.getRequestEntity().equals(requestEntity))
				.sorted(Comparator.comparing(Request::getCreatedAt))
				.findFirst();

		return req;
	}

	/**
	 * Average waiting time in minutes rounded to two decimal places for requests created in the last hour.
	 *
	 * @return
	 */
	public Optional<String> averageWaiting() {
		OptionalDouble optionalDouble = averageWaitingDouble();

		if (!optionalDouble.isPresent()) {
			return Optional.empty();
		}

		DecimalFormat df = new DecimalFormat("#.##");

		return Optional.of(df.format(optionalDouble.getAsDouble()));
	}

	public OptionalDouble averageWaitingDouble() {
		return getHandled().stream()
				.filter(r -> r.getHandledAt().isAfter(now().minusHours(1)))
				.mapToLong(Request::waitingTime)
				.average();
	}

	public OptionalDouble averageProcessingDouble() {
		return getHandled().stream()
				.filter(r -> r.getHandledAt().isAfter(now().minusHours(1)))
				.mapToLong(Request::processingTime)
				.average();
	}

	public Optional<String> averageProcessing() {
		OptionalDouble optionalDouble = averageProcessingDouble();

		if (!optionalDouble.isPresent()) {
			return Optional.empty();
		}

		DecimalFormat df = new DecimalFormat("#.##");

		return Optional.of(df.format(optionalDouble.getAsDouble()));
	}

	public Stream<User> activeAssistants() {
		return getArchived().stream()
				.filter(r -> r.getHandledAt().isAfter(now().minusHours(1)))
				.map(Request::getAssistant).filter(Objects::nonNull);
	}

	public Long activeAssistantsCount() {
		return activeAssistants().distinct().count();
	}

	public Long nrOfStudentsInQueue() {
		return getQueue().stream().map(Request::getRequestEntity).distinct().count();
	}

	@Cacheable("requestsHandledPerHour")
	public String requestsHandledPerHour() throws JsonProcessingException {
		return Request.requestsPerHour(this.getSlot().getOpensAt(), this.getSlot().getClosesAt(),
				this.getHandled());
	}

	@Cacheable("requestsTotalPerHour")
	public String requestsTotalPerHour() throws JsonProcessingException {
		return Request.requestsPerHour(this.getSlot().getOpensAt(), this.getSlot().getClosesAt(),
				this.getRequests());
	}

	@Cacheable("requestsApprovedPerHour")
	public String requestsApprovedPerHour() throws JsonProcessingException {
		return Request.requestsPerHour(this.getSlot().getOpensAt(), this.getSlot().getClosesAt(),
				this.getApproved());
	}

	@Cacheable("requestsRejectedPerHour")
	public String requestsRejectedPerHour() throws JsonProcessingException {
		return Request.requestsPerHour(this.getSlot().getOpensAt(), this.getSlot().getClosesAt(),
				this.getRejected());
	}

	/**
	 * Check if the given student is enqueued
	 *
	 * @param  entity
	 *
	 * @return
	 */
	public boolean isEnqueued(RequestEntity entity) {
		RequestEntity finalEntity = getEntityFor(entity);
		return getQueue().stream().anyMatch(r -> r.getRequestEntity().equals(finalEntity));
	}

	/**
	 * Takes a requestEntity (User) and returns the Group entity, if any else returns the user
	 *
	 * @param  requestEntity
	 *
	 * @return
	 */
	public RequestEntity getEntityFor(RequestEntity requestEntity) {
		if (getCourse().getHasGroups() && requestEntity instanceof User) {
			requestEntity = getCourse().getGroup((User) requestEntity);
		}
		return requestEntity;
	}

	/**
	 * Check if the given student is being processed
	 *
	 * @param  requestEntity
	 *
	 * @return
	 */
	public boolean isBeingProcessed(RequestEntity requestEntity) {
		return getRequests().stream()
				.filter(r -> r.getRequestEntity().equals(requestEntity))
				.anyMatch(Request::isProcessing);
	}

	/**
	 * Returns true if the lab is an online lab.
	 *
	 * @return true is online lab
	 */
	public boolean isOnline() {
		return communicationMethod.equals(CommunicationMethod.JITSI_MEET);
	}

	/**
	 * Returns true if a Jitsi link needs to be provided to the request entity.
	 *
	 * @return true is Jitsi link needed
	 */
	public boolean needsJitsiLink(RequestEntity requestEntity) {
		return isOnline() && isBeingProcessed(requestEntity);
	}

	/**
	 * Get requests for the given student
	 *
	 * @param  requestEntity
	 *
	 * @return
	 */
	public List<Request> requestsBy(RequestEntity requestEntity) {
		RequestEntity finalEntity = getEntityFor(requestEntity);
		return getRequests().stream()
				.filter(r -> r.getRequestEntity().equals(finalEntity))
				.sorted(Comparator.comparing(Request::getCreatedAt))
				.collect(Collectors.toList());
	}

	public Long countRequestsBy(RequestEntity requestEntity) {
		RequestEntity finalEntity = getEntityFor(requestEntity);
		return getRequests().stream()
				.filter(r -> r.getRequestEntity().equals(finalEntity))
				.count();
	}

	/**
	 * Get position of given student in queue. Assumes the student is enqueued.
	 *
	 * @param  requestEntity
	 *
	 * @return
	 */
	public String position(RequestEntity requestEntity) {
		int i = 1;

		for (Request request : getQueue()) {
			if (requestEntity.equals(request.getRequestEntity())) {
				if (request.getSlot() != null) {
					return formatPositionTimeSlot(request.getSlot());
				} else {
					return Integer.toString(i);
				}
			}
			i++;
		}
		return Integer.toString(i);
	}

	private String formatPositionTimeSlot(RequestSlot slot) {
		DateTimeFormatter formatterTime = DateTimeFormatter.ofPattern("HH:mm");
		DateTimeFormatter formatterDate = DateTimeFormatter.ofPattern("MMMM d yyyy");
		String start = slot.getOpensAt().format(formatterTime);
		String end = slot.getClosesAt().format(formatterTime);
		return slot.getOpensAt().format(formatterDate) + " at " + start + " - " + end;
	}

	public Boolean slotIsAvailable(RequestSlot slot) {
		LocalDateTime now = now();
		LocalDateTime twoMinutesGracePeriod = now.minusMinutes(2L);
		if (slot.getOpensAt().isBefore(twoMinutesGracePeriod)
				|| (slot.getOpensAt().isBefore(now) && slot.getClosesAt().isAfter(now))) {
			return false;
		}
		int numberOfRequestsOnSlot = getNumberOfRequestsOnSlot(slot);
		return numberOfRequestsOnSlot < capacity;
	}

	/**
	 * Gets the number of requests for a timeslot.
	 *
	 * @param  slot The timeslot whose number of requests is to be determined.
	 * @return      The number of requests on the timeslot.
	 */
	public int getNumberOfRequestsOnSlot(RequestSlot slot) {
		int numberOfRequestsOnSlot = 0;
		for (Request request : getQueue()) {
			if (request.getSlot() != null) {
				if (slot.getOpensAt().isEqual(request.getSlot().getOpensAt())) {
					numberOfRequestsOnSlot += 1;
				}
			}
		}
		return numberOfRequestsOnSlot;
	}

	/**
	 * Gets the requests on the specified slot.
	 *
	 * @param  slot The slot whose requests to get.
	 * @return      A list of requests.
	 */
	public List<Request> getRequestsOnSlot(RequestSlot slot) {
		return getQueue()
				.stream()
				.filter(request -> request.getSlot() != null)
				.filter(request -> request.getSlot().getOpensAt().isEqual(slot.getOpensAt()))
				.collect(Collectors.toList());
	}

	public void addAssignment(Assignment assignment) {
		assignments.add(assignment);

		if (!assignment.getLabs().contains(this)) {
			assignment.addLab(this);
		}
	}

	public boolean containsAssignment(Assignment assignment) {
		return assignments.contains(assignment);
	}

	public boolean containsAllowedRequestType(RequestType requestType) {
		return allowedRequestTypes.contains(requestType);
	}

	public boolean containsAllowedMentorGroup(FirstYearMentorGroup mentorGroup) {
		return allowedMentorGroups.contains(mentorGroup);
	}

	public boolean isOpen() {
		return slot.contains(now());
	}

	public boolean slotSelectionIsOpen() {
		return signOffIntervals && slotSelectionOpensAt != null && now().isAfter(slotSelectionOpensAt)
				&& now().isBefore(getSlot().getClosesAt());
	}

	public boolean isOpenOrSlotSelection() {
		return isOpen() || slotSelectionIsOpen();
	}

	public boolean display() {
		return slot.getClosesAt().plusMinutes(15).isAfter(now()) || isOpenOrSlotSelection();
	}

	public boolean hasRoom(Room room) {
		return rooms.contains(room);
	}

	/**
	 *
	 * @return all rooms used in the lab except for special placeholder rooms
	 */
	public List<Room> getFilteredRooms() {
		return rooms.stream().filter(r -> !r.getPlaceholder()).collect(Collectors.toList());
	}

	public void addRoom(Room room) {
		rooms.add(room);
	}

	public void addRequest(Request request) {
		requests.add(request);

		if (request.getLab() != this) {
			request.setLab(this);
		}
	}

	public boolean allAllowed() {
		return !allowWithoutMentorGroup && allowedMentorGroups.size() == 0;
	}

	/**
	 * Checks whether the user is allowed to enqueue in this lab.
	 *
	 * @param  user The user.
	 * @return      True if the user is allowed to enqueue in the lab.
	 */
	public boolean userAllowedForThisLab(User user) {
		List<FirstYearStudent> students = user.getFirstYearStudents();
		if (students == null) {
			return allowWithoutMentorGroup;
		}
		List<FirstYearStudent> activeStudents = students.stream()
				.filter(firstYearStudent -> firstYearStudent.getMentorGroup().isActive())
				.collect(Collectors.toList());
		if (activeStudents.isEmpty()) {
			logger.info("user has no first year mentor groups");
			return allowWithoutMentorGroup;
		} else {
			for (FirstYearStudent firstYearStudent : students) {
				if (containsAllowedMentorGroup(firstYearStudent.getMentorGroup())) {
					logger.info("Found an allowed mentorgroup");
					return true;
				}
			}
			return false;
		}
	}

	public String toReadableString() {
		LocalDateTime time = this.slot.getOpensAt();
		DateTimeFormatter formatter = DateTimeFormatter.ofPattern("HH:mm - " +
				"dd MMM uu");

		return "lab " + this.title + ": " + formatter.format(time);
	}

	/**
	 * Checks whether this group is allowed to enqueue in the lab. NB: at the moment this requires _all_
	 * members to be allowed
	 *
	 * @param  group The group that wants to enqueue.
	 * @return       True if the group is allowed to enqueue.
	 */
	public boolean groupAllowedForThisLab(Group group) {
		boolean allowed = true;
		for (User user : group.getMembers()) {
			allowed &= userAllowedForThisLab(user);
		}
		return allowed;
	}

	/**
	 * Checks whether the requestEntity is allowed to enqueue for this lab.
	 *
	 * @param  entity The entity.
	 * @return        True if the entity is allowed to enqueue for this lab.
	 */
	public boolean entityAllowedForThisLab(RequestEntity entity) {
		if (entity instanceof Group) {
			logger.info("Checking group access for: " + entity.getDisplayName());
			return groupAllowedForThisLab((Group) entity);
		} else if (entity instanceof User) {
			logger.info("Checking user access for: " + entity.getDisplayName());
			return userAllowedForThisLab((User) entity);
		}
		logger.info("Unknown entity: " + entity.getDisplayName());
		return false;
	}

	/**
	 * @param  requests Active requests
	 * @param  user     Active user currently working in the queue
	 * @return          Number of requests for this lab with status Pending plus nr of requests assigned to
	 *                  current user
	 */
	public Long getQueuedCount(List<Request> requests, User user) {
		List<Request> queuedForThisLab = requests.stream()
				.filter(request -> request.getLab().getId().equals(this.getId()))
				.filter(request -> request.getStatus().equals(Request.Status.PENDING)
						|| request.getStatus().equals(Request.Status.PICKED))
				.collect(Collectors.toList());
		if (isSignOffIntervals()) {
			LocalDateTime acceptInterval = getAcceptInterval();
			queuedForThisLab = queuedForThisLab.stream().filter(req -> req.getSlot() != null &&
					req.getSlot().getOpensAt().isBefore(acceptInterval))
					.collect(Collectors.toList());
		}

		return queuedForThisLab.size() + requests.stream()
				.filter(Request::isAssigned)
				.filter(request -> request.getAssistant().getId().equals(user.getId()))
				.filter(request -> request.getLab().getId().equals(this.getId()))
				.count();
	}

	/**
	 * This method gets the time from which a TA is able to accept a new request. If the student needs to go
	 * to the TA then he can accept up to 1 time slot earlier else only from 1 minute before the time slot.
	 *
	 * @return A time from when the TA can accept the request.
	 */
	public LocalDateTime getAcceptInterval() {
		if (this.getCommunicationMethod().equals(CommunicationMethod.STUDENT_VISIT_TA)) {
			return LocalDateTime.now().plusMinutes(this.intervalTime);
		} else {
			return LocalDateTime.now().plusMinutes(1L);
		}
	}

	@Override
	public String toString() {
		String returnString = "Lab " + getId();
		if (title != null) {
			returnString += ": " + title;
		}
		return returnString;
	}

	public String toSlug() {
		return "lab" + getId();
	}

}
