/*
 * Queue - A Queueing system that can be used to handle labs in higher education
 * Copyright (C) 2016-2020  Delft University of Technology
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <https://www.gnu.org/licenses/>.
 */
package nl.tudelft.ewi.queue.model;

import java.time.LocalDateTime;
import java.util.Objects;

import javax.persistence.*;

import lombok.Getter;
import lombok.NoArgsConstructor;
import lombok.Setter;

import org.json.JSONObject;
import org.springframework.format.annotation.DateTimeFormat;

import com.fasterxml.jackson.annotation.JsonIgnore;

@Entity
@Getter
@Setter
@NoArgsConstructor
public class Notification {
	public final static String TIME_FORMAT = "dd/MM/yyyy HH:mm";

	@Id
	@GeneratedValue(strategy = GenerationType.IDENTITY)
	private Long id;

	@ManyToOne
	@JsonIgnore
	private User user;

	@ManyToOne
	@JsonIgnore
	private Course course;

	private String title;

	private String message;

	private int ttl;

	private boolean read;

	private String url;

	@DateTimeFormat(pattern = TIME_FORMAT)
	private LocalDateTime createdAt;

	public Notification(User user, Course course, String title, String message, int ttl) {
		this.user = user;
		this.course = course;
		this.title = title;
		this.message = message;
		this.ttl = ttl;
		this.createdAt = LocalDateTime.now();
	}

	public Notification(User user, Course course, String title, String message, int ttl, String url) {
		this(user, course, title, message, ttl);
		this.url = url;
	}

	public String getUrl() {
		if (this.url == null) {
			return "/notification/" + id;
		}
		return this.url;
	}

	public boolean isUnread() {
		return !isRead();
	}

	public String toJSON() {
		return new JSONObject()
				.put("title", title)
				.put("message", message)
				.toString();
	}

	@Override
	public boolean equals(Object o) {
		if (this == o) {
			return true;
		}

		if (o == null || getClass() != o.getClass()) {
			return false;
		}

		Notification that = (Notification) o;

		return Objects.equals(id, that.id);
	}

	@Override
	public int hashCode() {
		return Objects.hash(id);
	}
}
