/*
 * Queue - A Queueing system that can be used to handle labs in higher education
 * Copyright (C) 2016-2020  Delft University of Technology
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <https://www.gnu.org/licenses/>.
 */
package nl.tudelft.ewi.queue.model;

import java.io.Serializable;
import java.sql.Timestamp;
import java.time.LocalDateTime;
import java.time.format.DateTimeFormatter;
import java.time.temporal.ChronoUnit;
import java.util.*;
import java.util.stream.Collectors;

import javax.persistence.*;
import javax.validation.constraints.NotNull;
import javax.validation.constraints.Size;

import nl.tudelft.ewi.queue.cqsr.Event;
import nl.tudelft.ewi.queue.cqsr.RecordsEvents;
import nl.tudelft.ewi.queue.views.View;

import org.apache.commons.lang.time.DateUtils;
import org.jooq.lambda.Seq;

import com.fasterxml.jackson.annotation.JsonIdentityInfo;
import com.fasterxml.jackson.annotation.JsonView;
import com.fasterxml.jackson.annotation.ObjectIdGenerators;
import com.fasterxml.jackson.core.JsonProcessingException;
import com.fasterxml.jackson.databind.ObjectMapper;

/**
 * A request that has been made by a {@link RequestEntity}, that the database representation of someone asking
 * for help. Part of this class stores it's data directly in the Request table. The lifecycle of a Request,
 * that is being: created, reassigned, forwarded, approved, etc; is done through event sourcing
 * {@link RequestEvent}. Every change in the requests lifecycle append a new event to this log.
 *
 * The entire log is then reran when the Request is recreated by hibernate through an {@link PostLoad} hook,
 * which fills in the remaining transient (not persisted) fields of this class.
 */
@Entity
@JsonIdentityInfo(generator = ObjectIdGenerators.PropertyGenerator.class, property = "id")
public class Request implements RecordsEvents<Request>, Serializable, Comparable<Request> {

	private static final long serialVersionUID = -6265456635668203715L;

	@Id
	@GeneratedValue(strategy = GenerationType.IDENTITY)
	private Long id;

	@ManyToOne
	@JoinColumn(name = "request_entity_id")
	@JsonView(View.Summary.class)
	private RequestEntity requestEntity;

	@NotNull
	@ManyToOne
	@JsonView(View.Summary.class)
	private Room room;

	private String jitsiRoom;

	@NotNull
	@JsonView(View.Summary.class)
	@OneToOne
	private RequestType requestType;

	@JsonView(View.Summary.class)
	@Size(max = 250)
	private String comment;

	@Column
	@Lob
	private String question;

	@JsonView(View.Summary.class)
	@Size(max = 250)
	private String feedback;

	@Column
	private Integer feedbackRating;

	@ManyToOne
	@JoinColumn(name = "assistant_id")
	@JsonView(View.Summary.class)
	private User assistant;

	@ManyToOne
	@JsonView(View.Summary.class)
	private Lab lab;

	@NotNull
	@ManyToOne
	@JsonView(View.Summary.class)
	private Assignment assignment;

	@OneToMany(cascade = { CascadeType.ALL }, mappedBy = "request")
	@JsonView(View.Summary.class)
	private List<RequestEvent> events = new ArrayList<>();

	@OneToOne(cascade = { CascadeType.ALL }, orphanRemoval = true)
	@JsonView(View.Summary.class)
	private RequestSlot slot;

	private Status status;

	@Transient
	private String reason;

	@Transient
	private String reasonForStudent;

	@Transient
	private LocalDateTime createdAt;

	@Transient
	private LocalDateTime revokedAt;

	@Transient
	private LocalDateTime assignedAt;

	@Transient
	private LocalDateTime processingAt;

	@Transient
	private LocalDateTime approvedAt;

	@Transient
	private LocalDateTime rejectedAt;

	@Transient
	private LocalDateTime forwardedAt;

	@Transient
	private LocalDateTime notFoundAt;

	@Transient
	private User forwardedTo;

	@Transient
	private Boolean leftFeedback;

	private static final String LABELS = "labels";
	private static final String DATA = "data";

	public Request() {
	}

	public Request(RequestEntity requestEntity, Assignment assignment, Room room, RequestType type,
			String comment, Lab lab) {
		this.requestEntity = requestEntity;
		this.assignment = assignment;
		this.room = room;
		this.requestType = type;
		this.comment = comment;
		this.lab = lab;

		this.requestCreatedEvent();
	}

	public Request(RequestEntity requestEntity, Assignment assignment, Room room,
			RequestType type, String comment, Lab lab, RequestSlot slot) {
		this.requestEntity = requestEntity;
		this.assignment = assignment;
		this.room = room;
		this.requestType = type;
		this.comment = comment;
		this.lab = lab;
		this.slot = slot;

		this.requestCreatedEvent();
	}

	public void requestCreatedEvent() {
		this.addEvent(new RequestCreatedEvent(this, LocalDateTime.now()));
		this.apply();
	}

	public static int getHour(Request request) {
		return request.getCreatedAt().getHour();
	}

	public Long getId() {
		return id;
	}

	public RequestEntity getRequestEntity() {
		return requestEntity;
	}

	public void setRequestEntity(RequestEntity requestEntity) {
		this.requestEntity = requestEntity;
	}

	public RequestType getRequestType() {
		return requestType;
	}

	public void setRequestType(RequestType type) {
		this.requestType = type;
	}

	public Room getRoom() {
		return room;
	}

	public void setRoom(Room room) {
		this.room = room;
	}

	public String getJitsiRoom() {
		return jitsiRoom;
	}

	/**
	 * Creates a name for a Jitsi room based on the name of the RequestEntity and the date.
	 */
	public void setJitsiRoom() {
		this.jitsiRoom = this.getRequestEntity().getDisplayName().replaceAll("\\s+", "") +
				LocalDateTime.now().format(DateTimeFormatter.ofPattern("ddmmHHMMss"));
	}

	public void setJitsiRoom(String jitsiRoom) {
		this.jitsiRoom = jitsiRoom;
	}

	/**
	 * Determines whether the Jitsi Room should be displayed for a user.
	 *
	 * @param  user the user
	 * @return      true if the Jitsi room should be displayed
	 */
	public boolean shouldDisplayJitsiRoom(User user) {
		return lab.isOnline() && ((assistant != null && assistant.equals(user))
				|| user.isAdmin()
				|| (user.isTeacher() && !user.teaches(getLab().getCourse())));
	}

	public Lab getLab() {
		return lab;
	}

	public void setLab(Lab lab) {
		this.lab = lab;

		if (!lab.getRequests().contains(this)) {
			lab.getRequests().add(this);
		}
	}

	public Assignment getAssignment() {
		return assignment;
	}

	public void setAssignment(Assignment assignment) {
		this.assignment = assignment;
	}

	public Status getStatus() {
		return status;
	}

	public Request setStatus(Status status) {
		this.status = status;

		return this;
	}

	public String getReason() {
		return reason;
	}

	public Request setReason(String reason) {
		this.reason = reason;

		return this;
	}

	public String getReasonForStudent() {
		return reasonForStudent;
	}

	public Request setReasonForStudent(String reasonForStudent) {
		this.reasonForStudent = reasonForStudent;

		return this;
	}

	public String getComment() {
		return comment;
	}

	public void setComment(String comment) {
		this.comment = comment;

	}

	public String getFeedback() {
		return feedback;
	}

	public void setFeedback(String feedback) {
		this.feedback = feedback;
	}

	public Boolean hasFeedback() {
		return feedback != null && !feedback.isEmpty();
	}

	public Boolean hasFeedbackRating() {
		return feedbackRating != null;
	}

	public LocalDateTime getCreatedAt() {
		return createdAt;
	}

	public Request setCreatedAt(LocalDateTime createdAt) {
		this.createdAt = createdAt;

		return this;
	}

	public LocalDateTime getRevokedAt() {
		return revokedAt;
	}

	public Request setRevokedAt(LocalDateTime revokedAt) {
		this.revokedAt = revokedAt;

		return this;
	}

	public LocalDateTime getAssignedAt() {
		return assignedAt;
	}

	public Request setAssignedAt(LocalDateTime assignedAt) {
		this.assignedAt = assignedAt;

		return this;
	}

	public LocalDateTime getProcessingAt() {
		return processingAt;
	}

	public Request setProcessingAt(LocalDateTime processingAt) {
		this.processingAt = processingAt;

		return this;
	}

	public LocalDateTime getApprovedAt() {
		return approvedAt;
	}

	public Request setApprovedAt(LocalDateTime approvedAt) {
		this.approvedAt = approvedAt;

		return this;
	}

	public LocalDateTime getRejectedAt() {
		return rejectedAt;
	}

	public LocalDateTime getNotFoundAt() {
		return notFoundAt;
	}

	public Request setRejectedAt(LocalDateTime rejectedAt) {
		this.rejectedAt = rejectedAt;

		return this;
	}

	public LocalDateTime getForwardedAt() {
		return forwardedAt;
	}

	public Request setForwardedAt(LocalDateTime forwardedAt) {
		this.forwardedAt = forwardedAt;

		return this;
	}

	public Request setNotFoundAt(LocalDateTime notFoundAt) {
		this.notFoundAt = notFoundAt;
		return this;
	}

	public User getForwardedTo() {
		return forwardedTo;
	}

	public Request setForwardedTo(User forwardedTo) {
		this.forwardedTo = forwardedTo;

		return this;
	}

	public RequestSlot getSlot() {
		return slot;
	}

	public void setSlot(RequestSlot slot) {
		this.slot = slot;
	}

	public String getSlotSentence() {
		DateTimeFormatter formatter = DateTimeFormatter.ofPattern("HH:mm");
		String start = slot.getOpensAt().format(formatter);
		String end = slot.getClosesAt().format(formatter);
		String ret = start + " - " + end;
		if (!DateUtils.isSameDay(Timestamp.valueOf(slot.getClosesAt()), new Date())) {
			DateTimeFormatter dateFormatter = DateTimeFormatter.ofPattern("d MMM uuuu");
			ret += " on " + slot.getClosesAt().format(dateFormatter);
		}
		return ret;
	}

	/**
	 * Gets the time at which the request was handled, i.e. approved, rejected or forwarded.
	 *
	 * @return The time as a LocalDateTime
	 */
	public LocalDateTime getHandledAt() {
		switch (getStatus()) {
			case APPROVED:
				return getApprovedAt();
			case REJECTED:
				return getRejectedAt();
			case FORWARDED:
				return getForwardedAt();
			case NOTFOUND:
				return getNotFoundAt();
			default:
				return null;
		}
	}

	/**
	 * Gets the time at which the request was archived, i.e. handled or revoked. Or null if it was not yet
	 * archived.
	 *
	 * @return The time this request was archived as a LocalDateTime.
	 */
	public LocalDateTime getArchivedAt() {
		switch (getStatus()) {
			case APPROVED:
				return getApprovedAt();
			case REJECTED:
				return getRejectedAt();
			case FORWARDED:
				return getForwardedAt();
			case NOTFOUND:
				return getNotFoundAt();
			case REVOKED:
			case CANCELLED:
				return getRevokedAt();
			default:
				return null;
		}
	}

	/**
	 * Gets the waiting time in minutes for this request, defined as moment of creation until moment of
	 * processing
	 *
	 * @return The waiting time as a Long
	 */
	public long waitingTime() {
		if (isPending()) {
			throw new IllegalStateException("No waiting time for a pending request");
		}

		LocalDateTime processTime = getProcessingAt();

		//If there is no processing time and is not pending than the request is already handled
		//so we take the handled time for calculation.
		if (processTime == null) {
			processTime = getHandledAt();
			if (processTime == null) {  // really?? There's no sensible answer anymore, return 42 seconds later
				processTime = getCreatedAt().plusSeconds(42);
			}
		}

		return ChronoUnit.MINUTES.between(getCreatedAt(), processTime);
	}

	/**
	 * Gets the processing time in minutes for this request defined as time processing starts until request is
	 * handled
	 *
	 * @return The processing time as a long.
	 */
	public long processingTime() {
		if (!isArchived()) {
			throw new IllegalStateException("No processing time for a pending request");
		}

		LocalDateTime processTime = getProcessingAt();

		// If there is no process time we dont know how long this request was processing for
		// so we take processing time from the moment the request gets created to the moment it is handled
		if (processTime == null) {
			processTime = getCreatedAt();
		}

		return ChronoUnit.MINUTES.between(processTime, getHandledAt());
	}

	/**
	 * Gets the assigned assistant for the request.
	 *
	 * @return null if no assistant has been assigned yet.
	 */
	public User getAssistant() {
		return assistant;
	}

	public Request setAssistant(User assistant) {
		this.assistant = assistant;
		return this;
	}

	/**
	 * Gets previous requests by the same student for the same assignment
	 *
	 * @return
	 */
	public List<Request> getPrevious() {
		return requestEntity.getRequests().stream()
				.filter(r -> r.getAssignment().equals(assignment))
				.filter(r -> r.getCreatedAt().isBefore(getCreatedAt()))
				.filter(Request::isHandled)
				.filter(r -> !r.equals(this))
				.collect(Collectors.toList());

	}

	@Override
	public List<RequestEvent> getEvents() {
		return events;
	}

	public List<RequestEvent> getPublicEvents() {
		return events.stream().filter(requestEvent -> !(requestEvent instanceof RequestPickedEvent))
				.collect(Collectors.toList());
	}

	@Override
	public void addEvent(Event<Request> event) {
		events.add((RequestEvent) event);
		event.apply(this);
	}

	@PostLoad
	public void postLoad() {
		this.apply();
	}

	@Override
	public Request apply() {
		return Seq.seq(getEvents()).foldLeft(this,
				(Request aggregate, Event<Request> event) -> event.apply(aggregate));
	}

	public boolean isPending() {
		return status == Status.PENDING || status == Status.PICKED;
	}

	public boolean isAssigned() {
		return status == Status.ASSIGNED;
	}

	public boolean isRevoked() {
		return status == Status.REVOKED;
	}

	public boolean isProcessing() {
		return status == Status.PROCESSING;
	}

	public boolean isApproved() {
		return status == Status.APPROVED;
	}

	public boolean isRejected() {
		return status == Status.REJECTED;
	}

	public boolean isForwarded() {
		return status == Status.FORWARDED;
	}

	public boolean isNotFound() {
		return status == Status.NOTFOUND;
	}

	public boolean isArchived() {
		return status == Status.APPROVED || status == Status.REJECTED || status == Status.FORWARDED
				|| status == Status.REVOKED || status == Status.CANCELLED || status == Status.NOTFOUND;
	}

	public boolean isHandled() {
		return status == Status.APPROVED || status == Status.REJECTED || status == Status.FORWARDED
				|| status == Status.NOTFOUND;
	}

	public boolean isPicked() {
		return status == Status.PICKED;
	}

	public boolean isQueued() {
		return isAssigned() || isPending();
	}

	@Override
	public String toString() {
		return "Request{" +
				"id=" + id +
				", status=" + status +
				", entity='" + requestEntity + '\'' +
				", assistant=" + assistant +
				", lab=" + lab +
				'}';
	}

	@Override
	public boolean equals(Object o) {
		if (this == o) {
			return true;
		}

		if (o == null || getClass() != o.getClass()) {
			return false;
		}

		Request request = (Request) o;

		return Objects.equals(id, request.id);
	}

	@Override
	public int hashCode() {
		return Objects.hash(id);
	}

	@Override
	public int compareTo(Request request) {
		if (this.slot != null && request.slot != null) {
			return this.slot.compareTo(request.slot);
		} else {
			if (this.createdAt.isBefore(request.getCreatedAt())) {
				return -1;
			} else if (this.createdAt.isAfter(request.getCreatedAt())) {
				return 1;
			}
		}
		return 0;
	}

	public String getQuestion() {
		return question;
	}

	public void setQuestion(String question) {
		this.question = question;
	}

	public Integer getFeedbackRating() {
		return feedbackRating;
	}

	public void setFeedbackRating(Integer feedbackRating) {
		this.feedbackRating = feedbackRating;
	}

	public enum Status {
		PICKED {
			@Override
			public String displayName() {
				return PENDING.displayName();
			}

			@Override
			public String toString() {
				return PENDING.toString();
			}
		},
		PENDING,
		ASSIGNED,
		PROCESSING,
		APPROVED,
		REJECTED,
		FORWARDED,
		REVOKED,
		CANCELLED,
		NOTFOUND {
			@Override
			public String displayName() {
				return "NOT FOUND";
			}
		};

		public String displayName() {
			return name();
		}
	}

	public String toSentence() {
		return requestType.toSentence(this, requestEntity, assignment);
	}

	public HashMap<String, String> toHashMap() {
		HashMap<String, String> requests = new HashMap<>();

		requests.put("id", this.getId().toString());
		requests.put("status", this.getStatus().toString());
		requests.put("sentence", this.toSentence());
		requests.put("labId", this.getLab().getId().toString());
		requests.put("lab", this.getLab().getTitle());
		requests.put("assignment", this.getAssignment().getName());
		requests.put("room", this.getRoom().getName());
		requests.put("course", this.getLab().getCourse().toString());

		if (this.getAssistant() != null)
			requests.put("assigned", this.getAssistant().getDisplayName());

		if (this.getHandledAt() != null)
			requests.put("handled", this.getHandledAt().toString());

		requests.put("display", this.displayRequest().toString());
		requests.put("type", this.getRequestType().getName());
		return requests;
	}

	public Boolean displayRequest() {
		if (!this.getLab().isSignOffIntervals()) {
			return true;
		}
		LocalDateTime interval = getLab().getAcceptInterval();
		if (slot == null) {
			System.err.println("Slot is null");
			return false;	// TODO I guess there is no point in trying to display this?
		}
		if (slot.getOpensAt() == null) {
			System.err.println("Slot opensAt is null");
			return false;	// TODO
		}
		return slot.getOpensAt().isBefore(interval);
	}

	public static String requestsPerHour(LocalDateTime start, LocalDateTime end, List<Request> requestList)
			throws JsonProcessingException {
		Map<String, ArrayList> graphData = requestsPerHourFor(start, end, requestList);

		ObjectMapper mapper = new ObjectMapper();
		ArrayList<Object> keyValues = new ArrayList<>();
		keyValues.add(graphData.get(LABELS));
		keyValues.add(graphData.get(DATA));
		return mapper.writeValueAsString(keyValues);
	}

	private static Map<String, ArrayList> requestsPerHourFor(LocalDateTime opensAt, LocalDateTime closesAt,
			List<Request> filteredRequests) {
		ArrayList<Long> handledPerHour = new ArrayList<>();
		DateTimeFormatter formatter = DateTimeFormatter.ofPattern("HH");
		ArrayList<String> hourLabels = new ArrayList<>();
		for (LocalDateTime date = opensAt; date.isBefore(closesAt); date = date.plusHours(1)) {
			Long nrOfRequestsThisHour = nrOfRequestsThisHour(date, filteredRequests);
			handledPerHour.add(nrOfRequestsThisHour);
			hourLabels.add(date.format(formatter));
		}
		HashMap<String, ArrayList> hashMap = new HashMap<>();
		hashMap.put("labels", hourLabels);
		hashMap.put("data", handledPerHour);
		return hashMap;
	}

	private static Long nrOfRequestsThisHour(LocalDateTime currentTime, List<Request> filteredRequests) {
		return filteredRequests.stream()
				.filter(r -> r.getCreatedAt().isAfter(currentTime) || r.getCreatedAt().isEqual(currentTime))
				.filter(r -> r.getCreatedAt().isBefore(currentTime.plusHours(1)))
				.count();
	}
}
