/*
 * Queue - A Queueing system that can be used to handle labs in higher education
 * Copyright (C) 2016-2020  Delft University of Technology
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <https://www.gnu.org/licenses/>.
 */
package nl.tudelft.ewi.queue.model;

import java.io.Serializable;
import java.util.List;
import java.util.stream.Collectors;

import javax.persistence.*;

@Entity
@Inheritance(strategy = InheritanceType.JOINED)
public abstract class RequestEntity implements Serializable {

	private static final long serialVersionUID = 4349260205347532487L;

	@Id
	@GeneratedValue(strategy = GenerationType.IDENTITY)
	Long id;

	private String displayName;

	@OneToMany(mappedBy = "requestEntity")
	private List<Request> requestsList;

	public RequestEntity() {
	}

	public RequestEntity(String displayName) {
		this.displayName = displayName;
	}

	public String getDisplayName() {
		return displayName;
	}

	public void setDisplayName(String displayName) {
		this.displayName = displayName;
	}

	@Override
	public String toString() {
		return getDisplayName();
	}

	public List<Request> getRequests() {
		return requestsList.stream().sorted((r1, r2) -> -r1.getCreatedAt().compareTo(r2.getCreatedAt()))
				.collect(Collectors.toList());
	}

	public Long getId() {
		return id;
	}

	public boolean participates(Course course) {
		if (this instanceof User) {
			User student = (User) this;
			return student.participates(course);
		} else {
			Group group = (Group) this;
			boolean allParticipate = true;
			for (User student : group.getMembers()) {
				if (!student.participates(course)) {
					allParticipate = false;
				}
			}
			return allParticipate;
		}
	}

	public boolean isGroup() {
		return this instanceof Group;
	}

	public boolean isUser() {
		return this instanceof User;
	}
}
