/*
 * Queue - A Queueing system that can be used to handle labs in higher education
 * Copyright (C) 2016-2020  Delft University of Technology
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <https://www.gnu.org/licenses/>.
 */
package nl.tudelft.ewi.queue.model;

import java.io.Serializable;

import javax.persistence.*;
import javax.validation.constraints.NotBlank;

import lombok.AllArgsConstructor;
import lombok.Data;
import lombok.NoArgsConstructor;
import nl.tudelft.ewi.queue.views.View;

import com.fasterxml.jackson.annotation.JsonView;

/**
 * Entity class representing the Room a lab can take place in.
 *
 * A room may be a placeholder, in which case it does not actually have a physical location bound to it. A
 * Room also always has a unique name used to identify it to Users.
 */
@Data
@NoArgsConstructor
@AllArgsConstructor
@Entity
public class Room implements Serializable, Comparable<Room> {

	private static final long serialVersionUID = 1092887696539958279L;

	@Id
	@GeneratedValue(strategy = GenerationType.IDENTITY)
	@JsonView(View.Summary.class)
	private Long id;

	@NotBlank
	@Column(unique = true)
	@JsonView(View.Summary.class)
	private String name;

	@JsonView(View.Summary.class)
	private String mapFilePath;

	@Column
	private Boolean placeholder = false;

	public Room(String name) {
		this.name = name;
	}

	public Room(String name, boolean placeholder) {
		this.name = name;
		this.placeholder = placeholder;
	}

	@Override
	public String toString() {
		return name;
	}

	@Override
	public int compareTo(Room room) {
		return name.compareTo(room.getName());
	}
}
