/*
 * Queue - A Queueing system that can be used to handle labs in higher education
 * Copyright (C) 2016-2020  Delft University of Technology
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <https://www.gnu.org/licenses/>.
 */
package nl.tudelft.ewi.queue.model;

import java.time.LocalDateTime;
import java.time.format.DateTimeFormatter;
import java.time.temporal.ChronoUnit;
import java.util.Objects;

import javax.persistence.*;
import javax.validation.constraints.NotNull;

import nl.tudelft.ewi.queue.helper.TimeHelper;
import nl.tudelft.ewi.queue.views.View;

import org.springframework.format.annotation.DateTimeFormat;

import com.fasterxml.jackson.annotation.JsonView;

@Entity
@Inheritance
public abstract class Slot implements Comparable<Slot> {
	public final static String TIME_FORMAT = "dd/MM/yyyy HH:mm";

	@Id
	@GeneratedValue(strategy = GenerationType.IDENTITY)
	private Long id;

	@DateTimeFormat(pattern = TIME_FORMAT)
	@NotNull
	@JsonView(View.Summary.class)
	private LocalDateTime opensAt;

	@DateTimeFormat(pattern = TIME_FORMAT)
	@JsonView(View.Summary.class)
	@NotNull
	private LocalDateTime closesAt;

	public Slot() {
	}

	public Slot(LocalDateTime opensAt, LocalDateTime closesAt) {
		this.opensAt = TimeHelper.setSecondsAndMsToZero(opensAt);
		this.closesAt = TimeHelper.setSecondsAndMsToZero(closesAt);
	}

	public Long getId() {
		return id;
	}

	public void setId(Long id) {
		this.id = id;
	}

	public LocalDateTime getOpensAt() {
		return opensAt;
	}

	public void setOpensAt(LocalDateTime opensAt) {
		this.opensAt = opensAt;
	}

	public LocalDateTime getClosesAt() {
		return TimeHelper.setSecondsAndMsToZero(closesAt);
	}

	public void setClosesAt(LocalDateTime closesAt) {

		this.closesAt = TimeHelper.setSecondsAndMsToZero(closesAt);
	}

	public boolean contains(LocalDateTime time) {
		return getOpensAt().isBefore(time) && getClosesAt().isAfter(time);
	}

	@Override
	public boolean equals(Object o) {
		if (this == o) {
			return true;
		}

		if (o == null || getClass() != o.getClass()) {
			return false;
		}

		Slot slot = (Slot) o;

		return Objects.equals(id, slot.id);
	}

	@Override
	public int hashCode() {
		return Objects.hash(id);
	}

	@Override
	public int compareTo(Slot that) {
		if (this.opensAt.isBefore(that.getOpensAt())) {
			return -1;
		} else if (that.getOpensAt().isBefore(this.opensAt)) {
			return 1;
		}
		return 0;
	}

	/**
	 * @return true if this lab slot is roughly today
	 */
	public boolean isTodayish() {
		LocalDateTime now = LocalDateTime.now();
		return now.isBefore(closesAt.plusDays(1)) && now.isAfter(opensAt.minusDays(1));
	}

	public Long durationOfSlot() {
		return this.opensAt.until(this.closesAt, ChronoUnit.MINUTES);
	}

	@Override
	public String toString() {
		DateTimeFormatter formatter = DateTimeFormatter.ofPattern("HH:mm");
		String start = formatter.format(opensAt);
		String end = formatter.format(closesAt);
		return start + " - " + end;
	}
}
