/*
 * Queue - A Queueing system that can be used to handle labs in higher education
 * Copyright (C) 2016-2020  Delft University of Technology
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <https://www.gnu.org/licenses/>.
 */
package nl.tudelft.ewi.queue.model;

import java.time.LocalDateTime;

import javax.persistence.Entity;
import javax.persistence.ManyToOne;
import javax.persistence.Transient;
import javax.validation.constraints.Size;

@Entity
public class StudentNotFoundEvent extends RequestEvent {

	private LocalDateTime notFoundAt;

	@Size(max = 250)
	private String reason;

	@Size(max = 250)
	private String reasonForStudent;

	@ManyToOne
	private User assistant;

	@Transient
	private String description;

	public StudentNotFoundEvent() {
		super();
	}

	public StudentNotFoundEvent(Request request, LocalDateTime localDateTime, User assistant) {
		super(request);
		this.notFoundAt = localDateTime;
		this.assistant = assistant;

		this.reason = this.getDescription();
		this.reasonForStudent = this.getDescription();
	}

	/**
	 * Get the description of the event based on the request's Communication Method.
	 *
	 * @return the description of the event
	 */
	@Override
	public String getDescription() {
		switch (request.getLab().getCommunicationMethod()) {
			case JITSI_MEET:
				return "Student did not connect on time";
			case STUDENT_VISIT_TA:
				return "Student did not show";
			case TA_VISIT_STUDENT:
				return "Could not find student to handle the request";
			default:
				return "Could not find each other";
		}
	}

	@Override
	public LocalDateTime getTimestamp() {
		return notFoundAt;
	}

	@Override
	public Request apply(Request request) {
		return request.setStatus(Request.Status.NOTFOUND)
				.setNotFoundAt(notFoundAt)
				.setReason(this.reason)
				.setReasonForStudent(this.reasonForStudent)
				.setAssistant(assistant);
	}

	@Override
	public String toString() {
		return "request-notfound";
	}
}
