/*
 * Queue - A Queueing system that can be used to handle labs in higher education
 * Copyright (C) 2016-2020  Delft University of Technology
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <https://www.gnu.org/licenses/>.
 */
package nl.tudelft.ewi.queue.model;

import java.util.ArrayList;
import java.util.List;
import java.util.Optional;
import java.util.stream.Collectors;

import javax.persistence.*;

import nl.tudelft.ewi.queue.service.UserService;
import nl.tudelft.ewi.queue.views.View;

import org.springframework.beans.factory.annotation.Autowired;

import com.fasterxml.jackson.annotation.JsonIgnore;
import com.fasterxml.jackson.annotation.JsonView;

@Entity
@Table(name = "person")	// Avoiding reserved database word user
public class User extends RequestEntity {

	private static final long serialVersionUID = 7273939479491656161L;

	@Column(unique = true)
	@JsonView(View.Summary.class)
	private String username;

	private String password;
	private String email;
	private int studentNumber;

	@Enumerated(EnumType.STRING)
	private DefaultRole defaultRole;

	@OneToMany(mappedBy = "user", cascade = { CascadeType.ALL })
	private List<Role> roles = new ArrayList<>();

	@OneToOne(cascade = { CascadeType.ALL })
	private Subscription subscription;

	// Students can become first year student again, hence the oneToMany
	@JsonIgnore
	@OneToMany(mappedBy = "user")
	private List<FirstYearStudent> firstYearStudents = new ArrayList<>();

	@OneToMany(mappedBy = "user")
	@OrderBy("createdAt desc")
	private List<Notification> notifications;

	@Autowired
	private static UserService userService;

	public User() {
	}

	public User(String username, String password, String displayName, String email, DefaultRole defaultRole,
			int studentNumber) {
		super(displayName);
		this.username = username;
		this.password = password;
		this.email = email;
		this.defaultRole = defaultRole;
		this.studentNumber = studentNumber;
	}

	public String getUsername() {
		return username;
	}

	public void setUsername(String username) {
		this.username = username;
	}

	public String getPassword() {
		return password;
	}

	public void setPassword(String password) {
		this.password = password;
	}

	public Subscription getSubscription() {
		return subscription;
	}

	public void setSubscription(Subscription subscription) {
		this.subscription = subscription;
	}

	public void setStudentNumber(int number) {
		this.studentNumber = number;
	}

	public NotificationList getNotifications() {
		return new NotificationList(notifications);
	}

	public boolean isOutdated() {
		return notifications.stream().anyMatch(Notification::isUnread);
	}

	public Integer notificationCount() {
		return notifications.size();
	}

	public DefaultRole getDefaultRole() {
		return defaultRole;
	}

	public void setDefaultRole(DefaultRole defaultRole) {
		this.defaultRole = defaultRole;
	}

	public boolean isAdmin() {
		return defaultRole == DefaultRole.ROLE_ADMIN;
	}

	public boolean isStudent() {
		return defaultRole == DefaultRole.ROLE_STUDENT;
	}

	public boolean isTeacher() {
		return defaultRole == DefaultRole.ROLE_TEACHER;
	}

	public List<FirstYearStudent> getFirstYearStudents() {
		return firstYearStudents;
	}

	public List<Role> getRoles() {
		return roles;
	}

	public void setRoles(List<Role> roles) {
		this.roles = roles;
	}

	public void addRole(Role role) {
		roles.add(role);

		if (!this.equals(role.getUser())) {
			role.setUser(this);
		}
	}

	/**
	 * Get this user's role in the given course
	 *
	 * @param  course
	 * @return
	 */
	public Optional<Role> getRole(Course course) {
		return getRoles().stream().filter(r -> r.getCourse().equals(course)).findAny();
	}

	/**
	 * Get the courses this user teaches
	 *
	 * @return
	 */
	public List<Course> getTeaches() {
		return getRoles().stream().filter(r -> (r instanceof Teacher)).map(Role::getCourse)
				.collect(Collectors.toList());
	}

	/**
	 * Check whether this user teaches the given course
	 *
	 * @param  course
	 * @return
	 */
	public boolean teaches(Course course) {
		return getTeaches().stream().anyMatch(c -> c.equals(course));
	}

	/**
	 * Get the courses this user manages
	 *
	 * @return
	 */
	public List<Course> getManages() {
		return getRoles().stream().filter(r -> (r instanceof Manager)).map(Role::getCourse)
				.collect(Collectors.toList());
	}

	/**
	 * Check whether this user manages the given course
	 *
	 * @param  course
	 * @return
	 */
	public boolean manages(Course course) {
		return getManages().stream().anyMatch(c -> c.equals(course));
	}

	/**
	 * Get the courses this user assists
	 *
	 * @return
	 */
	public List<Course> getAssists() {
		return getRoles().stream().filter(r -> (r instanceof Assistant)).map(Role::getCourse)
				.collect(Collectors.toList());
	}

	/**
	 * Check whether this user assists the given course
	 *
	 * @param  course
	 * @return
	 */
	public boolean assists(Course course) {
		return getAssists().stream().anyMatch(c -> c.equals(course));
	}

	/**
	 * Get the courses in which this user participates as a student
	 *
	 * @return
	 */
	public List<Course> getParticipates() {
		return getRoles().stream().filter(r -> (r instanceof Student)).map(Role::getCourse)
				.collect(Collectors.toList());
	}

	/**
	 * Check whether this user is enrolled for the given course
	 *
	 * @param  course
	 * @return
	 */
	@Override
	public boolean participates(Course course) {
		return getParticipates().stream().anyMatch(c -> c.equals(course));
	}

	/**
	 * Get user's requests ordered by creation time
	 *
	 * @return
	 */

	/*
	 * (non-Javadoc)
	 *
	 * @see java.lang.Object#hashCode()
	 */
	@Override
	public int hashCode() {
		final int prime = 31;
		int result = 1;
		result = prime * result + ((id == null) ? 0 : id.hashCode());
		result = prime * result + ((username == null) ? 0 : username.hashCode());
		return result;
	}

	/*
	 * (non-Javadoc)
	 *
	 * @see java.lang.Object#equals(java.lang.Object)
	 */
	@Override
	public boolean equals(Object obj) {
		if (this == obj) {
			return true;
		}
		if (obj == null) {
			return false;
		}
		if (!(obj instanceof User)) {
			return false;
		}
		User other = (User) obj;
		if (id == null) {
			if (other.id != null) {
				return false;
			}
		} else if (!id.equals(other.id)) {
			return false;
		}
		if (username == null) {
			if (other.username != null) {
				return false;
			}
		} else if (!username.equals(other.username)) {
			return false;
		}
		return true;
	}

	public String getEmail() {
		return email;
	}

	public void setEmail(String email) {
		this.email = email;
	}

	public int getStudentNumber() {
		return studentNumber;
	}

	/**
	 * @param  username hopefully a valid netID
	 * @return          a valid netID
	 */
	public static String guaranteeValidNetId(String username) {
		return userService.guaranteeValidNetId(username);
	}

	/*
	 * public void addFirstYearStudent(FirstYearStudent firstYearStudent) {
	 * this.firstYearStudents.add(firstYearStudent); }
	 */
}
