/*
 * Queue - A Queueing system that can be used to handle labs in higher education
 * Copyright (C) 2016-2020  Delft University of Technology
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <https://www.gnu.org/licenses/>.
 */
package nl.tudelft.ewi.queue.service;

import java.io.File;
import java.io.IOException;
import java.nio.file.Paths;
import java.util.List;

import javax.transaction.Transactional;

import nl.tudelft.ewi.queue.csv.EmptyCsvException;
import nl.tudelft.ewi.queue.csv.FirstYearStudentCsvHelper;
import nl.tudelft.ewi.queue.csv.InvalidCsvException;
import nl.tudelft.ewi.queue.model.FirstYearMentorGroup;
import nl.tudelft.ewi.queue.model.FirstYearStudent;
import nl.tudelft.ewi.queue.model.User;
import nl.tudelft.ewi.queue.repository.FirstYearMentorGroupRepository;
import nl.tudelft.ewi.queue.repository.FirstYearStudentRepository;
import nl.tudelft.ewi.queue.repository.UserRepository;

import org.apache.commons.io.FilenameUtils;
import org.apache.commons.lang.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.web.multipart.MultipartFile;

@Service
public class AdminService {
	// TODO this won't work when running locally; get from properties?
	public static final String QUEUE_MAPS_DIR = "/var/www/queue/maps/";

	private static final Logger logger = LoggerFactory.getLogger(AdminService.class);

	@Autowired
	private UserRepository ur;

	@Autowired
	private FirstYearMentorGroupRepository fymgr;

	@Autowired
	private FirstYearStudentRepository fysr;

	/**
	 * Marks all current mentor groups as inactive.
	 */
	private void markOldGroupsAsInactive() {
		List<FirstYearMentorGroup> groups = fymgr.findAll();
		for (FirstYearMentorGroup group : groups) {
			group.setActive(false);
		}
	}

	/**
	 * Links a student to a mentor group by their student number or net-id.
	 *
	 * @param netId       The student number/net-id of the student to be added.
	 * @param mentorGroup The mentor group the student should be added to.
	 */
	private void createFirstYearStudent(String netId, FirstYearMentorGroup mentorGroup) {
		User user;

		if (StringUtils.isNumeric(netId)) {
			// netId is actually a student number
			int studentNumber = Integer.parseInt(netId);
			user = ur.findByStudentNumber(studentNumber);
		} else {
			// netId is a net-id
			netId = User.guaranteeValidNetId(netId);
			user = ur.findByUsername(netId);
		}

		FirstYearStudent firstYearStudent = new FirstYearStudent(netId, mentorGroup);
		if (user != null) {
			firstYearStudent.setUser(user);
		}
		fysr.save(firstYearStudent);
	}

	/**
	 * Imports the given CSV file as a list of mentor groups and mentor group students.
	 *
	 * @param  csvFile             The CSV file to read into mentor groups.
	 * @return                     The amount of mentor groups saved in this method.
	 * @throws EmptyCsvException   when the CSV is empty.
	 * @throws InvalidCsvException when the CSV is not validly formed.
	 * @throws IOException         when some other error occurs when reading the CSV.
	 */
	@Transactional
	public int importCsv(MultipartFile csvFile) throws IOException, EmptyCsvException, InvalidCsvException {
		markOldGroupsAsInactive();

		int count = 0;
		List<FirstYearStudentCsvHelper> studentList = FirstYearStudentCsvHelper.readCsv(csvFile);
		for (FirstYearStudentCsvHelper helper : studentList) {
			FirstYearMentorGroup mentorGroup = fymgr.save(
					new FirstYearMentorGroup(helper.getMentorGroup()));

			createFirstYearStudent(helper.getNetId(), mentorGroup);

			count += 1;
		}
		return count;
	}

	/**
	 * Handles a file upload to the server by copying the given MultipartFile into the QUEUE_MAPS_DIR
	 * directory.
	 *
	 * @param  file        The file to upload and copy into the maps directory.
	 * @throws IOException when the file cannot be copied into the target directory.
	 */
	public void uploadFile(MultipartFile file) throws IOException {
		String fileName = file.getOriginalFilename();
		File uploadLocation = Paths.get(QUEUE_MAPS_DIR).toFile();
		File savedFile = Paths.get(QUEUE_MAPS_DIR, FilenameUtils.getName(fileName)).toFile();

		logger.warn("Uploading map with name {} in {}", fileName, QUEUE_MAPS_DIR);

		if (!uploadLocation.exists() && !uploadLocation.mkdirs()) {
			throw new IOException("Can not create path for storing maps: " + uploadLocation);
		}

		file.transferTo(savedFile);
	}
}
