/*
 * Queue - A Queueing system that can be used to handle labs in higher education
 * Copyright (C) 2016-2020  Delft University of Technology
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <https://www.gnu.org/licenses/>.
 */
package nl.tudelft.ewi.queue.service;

import java.io.IOException;
import java.time.LocalDateTime;
import java.util.*;

import nl.tudelft.ewi.queue.csv.EmptyCsvException;
import nl.tudelft.ewi.queue.csv.InvalidCsvException;
import nl.tudelft.ewi.queue.csv.UserCsvHelper;
import nl.tudelft.ewi.queue.model.*;
import nl.tudelft.ewi.queue.model.Course;
import nl.tudelft.ewi.queue.model.DefaultRole;
import nl.tudelft.ewi.queue.model.QRequest;
import nl.tudelft.ewi.queue.model.Request;
import nl.tudelft.ewi.queue.model.Student;
import nl.tudelft.ewi.queue.model.User;
import nl.tudelft.ewi.queue.repository.CourseRepository;
import nl.tudelft.ewi.queue.repository.RequestRepository;
import nl.tudelft.ewi.queue.repository.RoleRepository;
import nl.tudelft.ewi.queue.repository.UserRepository;

import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.web.multipart.MultipartFile;

import com.google.common.collect.Lists;
import com.querydsl.core.types.dsl.BooleanExpression;

@Service
public class CourseService {

	@Autowired
	private UserRepository userRepository;

	@Autowired
	private RequestRepository requestRepository;

	@Autowired
	private CourseRepository courseRepository;

	@Autowired
	private RoleRepository roleRepository;

	/**
	 * Enroll user for course
	 *
	 * @param user
	 * @param course
	 */
	public void enroll(User user, Course course) {
		user.addRole(new Student(user, course, LocalDateTime.now()));

		userRepository.save(user);
	}

	public List<Request> requestsWithFeedback(User assistant) {
		nl.tudelft.ewi.queue.model.QRequest request = QRequest.request;
		BooleanExpression hasFeedback = request.feedback.isNotNull();
		BooleanExpression handledByAssistant = request.assistant.id.eq(assistant.getId());

		return Lists.newArrayList(requestRepository.findAll(handledByAssistant.and(hasFeedback)));
	}

	public HashMap<Long, Integer> feedbackCount(List<User> assistants) {
		HashMap<Long, Integer> feedbackCount = new HashMap<>();
		for (User user : assistants) {
			int count = requestsWithFeedback(user).size();
			feedbackCount.put(user.getId(), count);
		}
		return feedbackCount;
	}

	/**
	 * Return count of requests handled per assistant per lab.
	 *
	 * @param  lab Lab for which the requests need to be counted.
	 * @return     HashMap which contains the amount of requests handled for every user.
	 */
	public HashMap<User, Integer> requestCountByAssistant(Lab lab) {
		HashMap<User, Integer> map = new HashMap<>();
		for (Request r : lab.getHandled()) {
			map.put(r.getAssistant(), map.getOrDefault(r.getAssistant(), 0) + 1);
		}
		return map;
	}

	/**
	 * Return map of individual requests handled for every lab of a course.
	 *
	 * @param  course Course for which requests need to be counted.
	 * @return        HashMap which contains the amount of requests handled for every user per course.
	 */
	public HashMap<Long, HashMap<User, Integer>> requestCountByAssistant(Course course) {
		HashMap<Long, HashMap<User, Integer>> map = new HashMap<>();
		map.put(0L, new HashMap<>());
		for (Lab lab : course.getLabs()) {
			HashMap<User, Integer> requests = requestCountByAssistant(lab);
			map.put(lab.getId(), requests);
			for (Map.Entry<User, Integer> e : requests.entrySet()) {
				map.get(0L).put(e.getKey(), map.get(0L).getOrDefault(e.getKey(), 0) + e.getValue());
			}
		}
		return map;
	}

	public void addCourseParticipants(MultipartFile csv, Course course)
			throws IOException, EmptyCsvException, InvalidCsvException {
		List<UserCsvHelper> users = UserCsvHelper.readCsv(csv);

		for (UserCsvHelper csvUser : users) {
			User staff = findOrCreateUser(csvUser.getNetId());
			Optional<Role> oldRole = staff.getRole(course);

			oldRole.ifPresent(r -> {
				staff.getRoles().remove(r);
				roleRepository.delete(r);
			});

			Role role = csvUser.getRole(staff, course);
			staff.addRole(role);
			course.addRole(role);
			userRepository.save(staff);
			courseRepository.save(course);
		}
	}

	private User findOrCreateUser(String netid) {
		User user = userRepository.findByUsername(netid + "@tudelft.nl");
		if (user == null) {
			user = new User(netid + "@tudelft.nl", "", netid, "",
					DefaultRole.ROLE_STUDENT, 0);
			userRepository.save(user);
		}
		return user;
	}

}
