/*
 * Queue - A Queueing system that can be used to handle labs in higher education
 * Copyright (C) 2016-2020  Delft University of Technology
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <https://www.gnu.org/licenses/>.
 */
package nl.tudelft.ewi.queue.service;

import java.io.IOException;
import java.util.List;

import nl.tudelft.ewi.queue.csv.BrightspaceCsvHelper;
import nl.tudelft.ewi.queue.csv.EmptyCsvException;
import nl.tudelft.ewi.queue.csv.InvalidCsvException;
import nl.tudelft.ewi.queue.model.Course;
import nl.tudelft.ewi.queue.model.DefaultRole;
import nl.tudelft.ewi.queue.model.Group;
import nl.tudelft.ewi.queue.model.User;
import nl.tudelft.ewi.queue.repository.GroupRepository;
import nl.tudelft.ewi.queue.repository.UserRepository;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.web.multipart.MultipartFile;

@Service
public class GroupService {
	private static final Logger logger = LoggerFactory.getLogger(GroupService.class);
	@Autowired
	private UserRepository userRepository;
	@Autowired
	private GroupRepository groupRepository;
	@Autowired
	private CourseService courseService;

	private void deleteOldGroups(Course course) {
		course.getGroups().removeAll(course.getGroups());
	}

	public void importCsv(Course course, MultipartFile csvFile)
			throws IOException, EmptyCsvException, InvalidCsvException {
		deleteOldGroups(course);

		List<BrightspaceCsvHelper> csvLines = BrightspaceCsvHelper.readCsv(csvFile);
		for (BrightspaceCsvHelper csvLine : csvLines) {
			User student = findOrCreateUser(csvLine);
			if (!course.isEnrolled(student)) {
				logger.info("Enrolling unenrolled student: " + student.getUsername());
				courseService.enroll(student, course);
			}
			Group group = findOrCreateGroup(csvLine, student, course);
			if (!course.isGroupOfThisCourse(group)) {
				logger.info("Adding group to course: " + group.getDisplayName());
				course.addGroup(group);
			}
		}
		//        groupRepository.save(course.getGroups());
	}

	private User findOrCreateUser(BrightspaceCsvHelper student) {
		Boolean exists = userRepository.existsByUsername(student.getNetId());
		User user;

		if (exists) {
			logger.info("Student found in DB");
			user = userRepository.findByUsername(student.getNetId());
		} else {
			logger.info("unknown student: " + student.getNetId());
			String displayName = student.getFirstName() + " " + student.getLastName();
			DefaultRole role = DefaultRole.ROLE_STUDENT;
			user = new User(student.getNetId(), "", displayName, student.getEmail(), role,
					student.getStudentNumber());
			userRepository.save(user);
		}

		/*
		 * if (firstYearStudentRepository.existsByNetId(user.getUsername())) {
		 * logger.info("Student has first year mentor groups"); FirstYearStudent firstYearStudent =
		 * firstYearStudentRepository.findByNetId(user.getUsername()); firstYearStudent.setUser(user);
		 * user.addFirstYearStudent(firstYearStudent); userRepository.save(user);
		 * firstYearStudentRepository.save(firstYearStudent); }
		 */
		return user;
	}

	private Group findOrCreateGroup(BrightspaceCsvHelper student, User user, Course course) {
		Boolean exists = groupRepository.existsByDisplayNameAndCourse(student.getGroupName(), course);
		Group group;
		if (exists) {
			group = groupRepository.findByDisplayNameAndCourse(student.getGroupName(), course);
		} else {
			group = new Group(student.getGroupName(), course);
		}
		group.addMember(user);
		groupRepository.save(group);
		return group;
	}

}
