/*
 * Queue - A Queueing system that can be used to handle labs in higher education
 * Copyright (C) 2016-2020  Delft University of Technology
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <https://www.gnu.org/licenses/>.
 */
package nl.tudelft.ewi.queue.service;

import java.time.LocalDateTime;
import java.time.temporal.ChronoUnit;
import java.util.ArrayList;
import java.util.List;
import java.util.Optional;
import java.util.Random;
import java.util.stream.Collectors;

import javax.persistence.EntityNotFoundException;

import nl.tudelft.ewi.queue.exception.AlreadyEnqueuedException;
import nl.tudelft.ewi.queue.exception.InvalidSlotException;
import nl.tudelft.ewi.queue.helper.EmailHelper;
import nl.tudelft.ewi.queue.model.*;
import nl.tudelft.ewi.queue.repository.*;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.security.access.AccessDeniedException;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

@Service
public class LabService {
	private static final Logger logger = LoggerFactory.getLogger(LabService.class);

	@Autowired
	private RequestService requestService;

	@Autowired
	private RequestRepository requestRepository;

	@Autowired
	private NotificationService notificationService;

	@Autowired
	private LabRepository labRepository;

	@Autowired
	private RequestTypeRepository requestTypeRepository;

	@Autowired
	private AssignmentRepository assignmentRepository;

	@Autowired
	private CourseRepository courseRepository;

	@Autowired
	private FirstYearMentorGroupRepository firstYearMentorGroupRepository;

	@Autowired
	private RoomRepository roomRepository;

	/**
	 * Enqueue student for lab. An exception is thrown if the lab is closed, the student is not enrolled, or
	 * if the student is already enqueued.
	 *
	 * @param request
	 */
	public void enqueue(Request request) {
		if (!request.getLab().isSignOffIntervals() && !request.getLab().isOpenOrSlotSelection()) {
			throw new AccessDeniedException("Enqueueing is only possible if the lab is open.");
		}

		if (!request.getRequestEntity().participates(request.getLab().getCourse())) {
			throw new AccessDeniedException("You need to be enrolled in the course for this lab.");
		}
		if (!request.getLab().isFeedbackLab()) {
			if (request.getLab().isEnqueued(request.getRequestEntity())) {
				throw new AlreadyEnqueuedException("Student is already enqueued.");
			}
		}

		if (request.getSlot() != null) {
			if (ChronoUnit.MINUTES.between(request.getSlot().getOpensAt(),
					request.getSlot().getClosesAt()) > request.getLab().getIntervalTime()) {
				throw new InvalidSlotException("Your interval is too long.");
			}
			if (request.getSlot().getOpensAt().isBefore(request.getLab().getSlot().getOpensAt())) {
				throw new InvalidSlotException("Your request interval is before the lab opens.");
			}
			if (!request.getLab().slotIsAvailable(request.getSlot())) {
				throw new InvalidSlotException("This slot is no longer available.");
			}
		}
		request.requestCreatedEvent();
		requestRepository.save(request);
		// only send a notification for a request that is not with an interval
		if (!request.getLab().isSignOffIntervals()) {
			createAndSendNotifications(request);
		}

	}

	public void createAndSendNotifications(Request request) {
		Lab lab = request.getLab();
		Course course = courseRepository.findById(lab.getCourse().getId()).orElseThrow();
		for (User assistant : course.getPrivileged()) {
			Notification notification = new Notification(assistant, course, "New request",
					"New request for " + lab + " in course " + course,
					100);
			notificationService.sendPushNotification(assistant, notification);
			notificationService.sendRequestTableUpdate(assistant, request);
		}
	}

	/**
	 * Given a lab, repeats that lab for the repeatAmount of weeks by setting the slot plus 1*x weeks per new
	 * lab
	 *
	 * @param repeatAmount number of times to repeat a lab
	 * @param lab          the lab that should be repeated
	 */
	public void repeatLabFor(int repeatAmount, Lab lab) {
		LocalDateTime slotOpen = lab.getSlot().getOpensAt();
		LocalDateTime slotClose = lab.getSlot().getClosesAt();
		LocalDateTime slotSelectionOpensAt = lab.getSlotSelectionOpensAt();
		for (int i = 0; i < repeatAmount; i++) {
			slotOpen = slotOpen.plusWeeks(1);
			slotClose = slotClose.plusWeeks(1);
			LabSlot nextSlot = new LabSlot(slotOpen, slotClose);
			List<Room> roomClone = new ArrayList<>(lab.getRooms());
			Lab clone = new Lab(lab.getCourse(), nextSlot, roomClone, lab.isFeedbackLab());

			if (lab.isSignOffIntervals()) {
				slotSelectionOpensAt = slotSelectionOpensAt.plusWeeks(1);
				clone.setSignOffIntervals(true);
				clone.setIntervalTime(lab.getIntervalTime());
				clone.setCapacity(lab.getCapacity());
				clone.setSlotSelectionOpensAt(slotSelectionOpensAt);
			}
			clone.setTitle(lab.getTitle());
			clone.setAssignments(copyAssignments(lab));
			clone.setCommunicationMethod(lab.getCommunicationMethod());
			clone.setAllowedRequestTypes(copyRequestTypes(lab));
			clone.setAllowedMentorGroups(copyAllowedMentorGroups(lab));
			clone.setAllowWithoutMentorGroup(lab.isAllowWithoutMentorGroup());

			labRepository.save(clone);
		}
	}

	public void updateLab(Long id, Lab labData) {
		Lab lab = getLab(id);

		lab.setTitle(labData.getTitle());
		lab.setCommunicationMethod(labData.getCommunicationMethod());
		lab.setSlot(labData.getSlot());
		lab.setRooms(labData.getRooms());
		lab.setAssignments(labData.getAssignments());
		lab.setCapacity(labData.getCapacity());
		lab.setIntervalTime(labData.getIntervalTime());
		lab.setAllowedRequestTypes(labData.getAllowedRequestTypes());
		lab.setSlotSelectionOpensAt(labData.getSlotSelectionOpensAt());
		lab.setAllowedMentorGroups(labData.getAllowedMentorGroups());
		lab.setAllowWithoutMentorGroup(labData.isAllowWithoutMentorGroup());

		labRepository.save(lab);
	}

	public void saveLab(Course course, Lab lab, Optional<Integer> weekRepeat) {
		lab.setCourse(course);

		if (lab.isSignOffIntervals()) {
			Room tbd = roomRepository.findByName("To be determined");
			if (tbd != null) {
				lab.addRoom(tbd);
			}
		}

		// In case we have an exam lab we need to schedule the picking of students.

		for (Assignment assignment : lab.getAssignments()) {
			if (assignment != null) {
				assignment.addLab(lab);
			}
		}
		weekRepeat.ifPresent(wr -> repeatLabFor(wr, lab));
		labRepository.save(lab);
	}

	/**
	 * No longer used? private ArrayList<Room> copyRooms(Lab lab) { ArrayList<Room> copiedRooms = new
	 * ArrayList<>(); for (Room room : lab.getRooms()) { copiedRooms.add(new Room(room.getName())); } return
	 * copiedRooms; }
	 */

	private ArrayList<Assignment> copyAssignments(Lab lab) {
		ArrayList<Assignment> copiedAssignments = new ArrayList<>();
		for (Assignment assignment : lab.getAssignments()) {
			if (assignment != null) {
				copiedAssignments.add(assignmentRepository.findById(assignment.getId()).orElseThrow());
			}
		}
		return copiedAssignments;
	}

	private ArrayList<FirstYearMentorGroup> copyAllowedMentorGroups(Lab lab) {
		ArrayList<FirstYearMentorGroup> mentorGroupsCopy = new ArrayList<>();
		for (FirstYearMentorGroup mentorGroup : lab.getAllowedMentorGroups()) {
			if (mentorGroup != null) {
				mentorGroupsCopy
						.add(firstYearMentorGroupRepository.findById(mentorGroup.getId()).orElseThrow());
			}
		}
		return mentorGroupsCopy;
	}

	private ArrayList<RequestType> copyRequestTypes(Lab lab) {
		ArrayList<RequestType> requestTypeCopy = new ArrayList<>();
		for (RequestType requestType : lab.getAllowedRequestTypes()) {
			if (requestType != null) {
				requestTypeCopy.add(requestTypeRepository.findById(requestType.getId()).orElseThrow());
			}
		}
		return requestTypeCopy;
	}

	public ArrayList<RequestSlot> getIntervalsForLab(Lab lab) {
		assert lab.isSignOffIntervals();
		Long intervalTimeMinutes = lab.getIntervalTime();
		ArrayList<RequestSlot> intervals = new ArrayList<>();
		LocalDateTime startTime = lab.getSlot().getOpensAt();
		LocalDateTime endTime = lab.getSlot().getClosesAt();

		for (LocalDateTime dateTime = startTime; dateTime
				.isBefore(endTime); dateTime = dateTime.plusMinutes(intervalTimeMinutes)) {
			RequestSlot slot = new RequestSlot(dateTime, dateTime.plusMinutes(intervalTimeMinutes));
			slot.setAvailable(lab.slotIsAvailable(slot));
			intervals.add(slot);
		}
		return intervals;
	}

	public Lab getLab(Long id) {
		return labRepository.findById(id)
				.orElseThrow(() -> new EntityNotFoundException("Entity was not found"));
	}

	public Assignment getAssignment(Long id) {
		return assignmentRepository.findById(id)
				.orElseThrow(() -> new EntityNotFoundException("Entity was not found"));
	}

	public Course getCourse(Long id) {
		return courseRepository.findById(id)
				.orElseThrow(() -> new EntityNotFoundException("Entity was not found"));
	}

	/**
	 * Picks a random timeslot in a lab that the requestEntity can see in which they can sign off the
	 * assignment and enqueues them in it. A requestEntity can be for example a User or a Group.
	 *
	 * @param  assignment The assignment.
	 * @param  entity     The requestEntity.
	 * @return            An optional with the lab if the user was enqueued, and an empty optional otherwise.
	 */
	public Optional<Lab> randomEnqueue(Assignment assignment, RequestEntity entity) {

		logger.error("******** Entering random enqueue ******");

		List<Lab> labsWithAssignment = labRepository.findAll()
				.stream()
				.filter(l -> l.getAssignments().contains(assignment)).collect(Collectors.toList());

		logger.error("** labs with assignment: {}", labsWithAssignment.size());

		List<Lab> labsWithSlot = labsWithAssignment.stream()
				.filter(l -> getIntervalsForLab(l)
						.stream()
						.anyMatch(l::slotIsAvailable))
				.collect(Collectors.toList());

		logger.error("** labs with assignment & slots: {}", labsWithSlot.size());

		List<Lab> labs = labsWithSlot.stream().filter(l -> l.entityAllowedForThisLab(entity))
				.collect(Collectors.toList());

		logger.error("** labs with assignment, slots and allowed: {}", labs.size());

		Random r = new Random();
		int labSize = labs.size();
		if (labSize > 0) {
			Optional<Lab> lab = Optional.ofNullable(labs.get(r.nextInt(labSize)));

			if (lab.isPresent()) {

				Optional<Request> prevReq = lab.get().getPendingRequest(entity);
				if (prevReq.isPresent()) {
					if (prevReq.get().getAssignment().equals(assignment)) {
						requestService.revoke(prevReq.get());
					}
				}

				List<RequestSlot> slots = getIntervalsForLab(lab.get()).stream()
						.filter(lab.get()::slotIsAvailable).collect(Collectors.toList());

				Optional<RequestSlot> slot = Optional.ofNullable(slots.get(r.nextInt(slots.size())));

				if (slot.isPresent()) {
					logger.error("Slot found");

					Request request = new Request(entity, assignment, lab.get().getRooms().get(0),
							requestTypeRepository.findByName("Submission"), "", lab.get(), slot.get());

					slot.get().setRequest(request);
					logger.error("request created");

					//TODO: do we want to unenqueue a student if they are already enqueued for this timeslot?
					//TODO: students can potentially have multiple assignments they can sign of in a lab -
					// then they should also be able to enqueue for multiple slots?
					enqueue(request);
					logger.error("request enqueued");

					if (entity instanceof User) {
						EmailHelper.getInstance().notifyStudentOfTimeSlot((User) entity,
								slot.get(), assignment);
					} else if (entity instanceof Group) {
						EmailHelper.getInstance().notifyGroupOfTimeSlot((Group) entity,
								slot.get(), assignment);
					}

					return lab;
				} else {
					logger.error("There were no free slots.");
				}
			} else {
				logger.error("Lab not present");
			}
		} else {
			logger.error("The mentorgroup this student belongs to has no labs.");
		}
		return Optional.empty();
	}

	@Transactional
	public void closeEnqueue(Long id, boolean close) {
		labRepository.findByIdOrThrow(id).setEnqueueClosed(close);
	}
}
