/*
 * Queue - A Queueing system that can be used to handle labs in higher education
 * Copyright (C) 2016-2020  Delft University of Technology
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <https://www.gnu.org/licenses/>.
 */
package nl.tudelft.ewi.queue.service;

import java.util.HashMap;
import java.util.Map;

import nl.martijndwars.webpush.PushService;
import nl.tudelft.ewi.queue.factory.NotificationFactory;
import nl.tudelft.ewi.queue.model.*;

import org.hibernate.Hibernate;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.messaging.handler.annotation.MessageMapping;
import org.springframework.messaging.simp.SimpMessageSendingOperations;
import org.springframework.scheduling.annotation.Async;
import org.springframework.stereotype.Service;

@Service
public class NotificationService {

	@Autowired
	private SimpMessageSendingOperations messagingTemplate;

	@Autowired
	private PushService pushService;

	@Autowired
	private NotificationFactory notificationFactory;

	@Async
	public void sendPushNotification(User receiver, Notification notification) {
		// Update UI through WebSocket
		HashMap<String, Object> pushNotification = new HashMap<>();
		pushNotification.put("notification", notification);

		sendWebSocketMessage(receiver, pushNotification);

		// Send push notification
		Subscription subscription = receiver.getSubscription();

		if (null != subscription) {
			try {
				pushService.sendAsync(
						notificationFactory.fromSubscription(subscription, notification.toJSON().getBytes()));
			} catch (Exception e) {
				e.printStackTrace();
			}
		}
	}

	public void sendRequestTableUpdateForAllAssistants(Request request) {
		for (User assistant : request.getLab().getCourse().getPrivileged()) {
			sendRequestTableUpdate(assistant, request);
		}
	}

	@Async
	public void sendRequestTableUpdate(User assistant, Request request) {
		Hibernate.initialize(request);
		if (request.getAssignment().allowedToHandleAssignment(assistant)) {
			HashMap<String, Object> requestTableUpdate = new HashMap<>();
			requestTableUpdate.put("request", request.toHashMap());

			sendWebSocketMessage(assistant, requestTableUpdate);
		}
	}

	/**
	 * Updates number which displays the student's position in the queue.
	 *
	 * @param student the student whose page is being updated
	 * @param lab     the lab for which the position is being moved
	 */
	@Async
	public void sendQueuePositionUpdate(User student, Lab lab) {
		HashMap<String, Object> queuePositionUpdate = new HashMap<>();
		queuePositionUpdate.put("position", lab.position(student));

		sendWebSocketMessage(student, queuePositionUpdate);
	}

	/**
	 * Notifies the student that their request is being processed.
	 *
	 * @param student the student to notify
	 */
	@Async
	public void sendRequestProcessingUpdate(User student) {
		HashMap<String, Object> queuePositionUpdate = new HashMap<>();
		queuePositionUpdate.put("position", "0");

		sendWebSocketMessage(student, queuePositionUpdate);
	}

	@MessageMapping("/notifications")
	private void sendWebSocketMessage(User receiver, HashMap<String, Object> message) {
		messagingTemplate.convertAndSendToUser(receiver.getUsername(), "/queue/notifications", message);
	}

	/**
	 * Creates a notification asking students to provide feedback about their TA. If the RequestEntity has
	 * several members, multiple notifications are created.
	 *
	 * @param  entity  The request entity which should receive the notification
	 * @param  request The request which should be linked to the notification.
	 * @return         Returns a map with User and Notification to show which notification should be send to
	 *                 which user.
	 */
	public Map<User, Notification> createFeedbackNotification(RequestEntity entity, Request request) {
		String url = "/request/" + request.getId();
		Map<User, Notification> notifications = new HashMap<>();
		if (entity.isGroup()) {
			for (User user : ((Group) entity).getMembers()) {
				notifications.put(user, new Notification(user, request.getLab().getCourse(),
						"Please provide feedback",
						"Please provide feedback for your TA.", 1000, url));
			}
		} else {
			notifications.put((User) entity, new Notification((User) entity, request.getLab().getCourse(),
					"Please provide feedback",
					"Please provide feedback for your TA.", 1000, url));
		}
		return notifications;
	}

	public Map<User, Notification> createNotFoundNotification(RequestEntity requestEntity, Request request) {
		Map<User, Notification> notifications = new HashMap<>();
		if (requestEntity.isGroup()) {
			for (User user : ((Group) requestEntity).getMembers()) {
				notifications.put(user, new Notification(user,
						request.getLab().getCourse(), "Not Found",
						"The TA was not able to find you. Please check if you filled in the correct room!",
						100));
			}
		} else {
			notifications.put((User) requestEntity, new Notification((User) requestEntity,
					request.getLab().getCourse(), "Not Found",
					"The TA was not able to find you. Please check if you filled in the correct room! Remember to respond to the TA when he/she calls your name!",
					100));
		}
		return notifications;
	}

}
