/*
 * Queue - A Queueing system that can be used to handle labs in higher education
 * Copyright (C) 2016-2020  Delft University of Technology
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <https://www.gnu.org/licenses/>.
 */
package nl.tudelft.ewi.queue.service;

import java.util.List;

import javax.persistence.EntityNotFoundException;

import nl.tudelft.ewi.queue.model.*;
import nl.tudelft.ewi.queue.repository.*;

import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

@Service
public class PermissionService {

	@Autowired
	private UserRepository userRepository;

	@Autowired
	private RoleRepository roleRepository;

	@Autowired
	private CourseRepository courseRepository;

	@Autowired
	private LabRepository labRepository;

	@Autowired
	private NotificationRepository notificationRepository;

	@Autowired
	private RequestRepository requestRepository;

	public boolean isAdmin(Object principal) {
		User user = getUser(principal);
		return user.isAdmin();
	}

	public boolean canViewCourse(Object principal, Long id) {
		User user = getUser(principal);

		Course course = courseRepository.findByIdOrThrow(id);

		return user.isAdmin() ||
				(user.participates(course) && !course.getIsArchived()) ||
				user.assists(course) ||
				user.manages(course) ||
				user.teaches(course);
	}

	/**
	 * Students, assistants and managers can always leave a course. A teacher can leave the course if he is
	 * not the last teacher.
	 */
	public boolean canLeaveCourse(Object principal, Long id) {
		User user = getUser(principal);

		Course course = courseRepository.findByIdOrThrow(id);

		return user.participates(course) ||
				user.assists(course) ||
				user.manages(course) ||
				(user.teaches(course) && course.getTeachers().size() > 1);
	}

	public boolean canManageParticipants(Object principal, Long courseId) {
		User user = getUser(principal);

		Course course = courseRepository.findByIdOrThrow(courseId);

		return user.isAdmin() ||
				user.teaches(course) ||
				user.manages(course);
	}

	public boolean canManageTeachers(Object principal, Long courseId) {
		User user = getUser(principal);

		Course course = courseRepository.findByIdOrThrow(courseId);

		return user.isAdmin() ||
				user.teaches(course);
	}

	public boolean canUpdateParticipant(Object principal, Long courseId, Long participantId) {
		User user = getUser(principal);
		User participant = getRole(participantId).getUser();

		Course course = courseRepository.findByIdOrThrow(courseId);

		return user.isAdmin() ||
				user.teaches(course) ||
				user.manages(course) && !participant.teaches(course) ||
				participant.getId().equals(user.getId());
	}

	public boolean canAddParticipant(Object principal, Long courseId, String participantRole) {
		User user = getUser(principal);

		Course course = courseRepository.findByIdOrThrow(courseId);

		return user.isAdmin() ||
				user.teaches(course) ||
				user.manages(course) && !"teacher".equals(participantRole);
	}

	public boolean canViewStatus(Object principal, Long id) {
		User user = getUser(principal);
		Course course = courseRepository.findByIdOrThrow(id);

		return user.isAdmin() ||
				user.manages(course) ||
				user.teaches(course);
	}

	public boolean canManageAssignments(Object principal, Long id) {
		User user = getUser(principal);

		Course course = courseRepository.findByIdOrThrow(id);

		return user.isAdmin() ||
				user.manages(course) ||
				user.teaches(course);
	}

	public boolean canEditCourse(Object principal, Long id) {
		User user = getUser(principal);

		Course course = courseRepository.findByIdOrThrow(id);

		return user.isAdmin() ||
				user.teaches(course);
	}

	public boolean canExportLab(Object principal, Long id) {
		User user = getUser(principal);
		Lab lab = labRepository.findByIdOrThrow(id);
		Course course = lab.getCourse();

		return user.isAdmin() ||
				user.teaches(course);
	}

	public boolean canRemoveCourse(Object principal, Long id) {
		User user = getUser(principal);

		Course course = courseRepository.findByIdOrThrow(id);

		return user.isAdmin() ||
				user.teaches(course);
	}

	public boolean canEnrollCourse(Object principal, Long id) {
		User user = getUser(principal);

		Course course = courseRepository.findByIdOrThrow(id);

		return !user.participates(course) &&
				!user.assists(course) &&
				!user.manages(course) &&
				!user.teaches(course) &&
				!course.getIsArchived();
	}

	public boolean canViewLab(Object principal, Long id) {
		User user = getUser(principal);

		Lab lab = getLab(id);

		Course course = lab.getCourse();

		return user.isAdmin() ||
				user.participates(course) ||
				user.assists(course) ||
				user.manages(course) ||
				user.teaches(course);
	}

	/**
	 * A user can enqueue for a lab if: - he is enrolled for the lab's course (i.e. 'participates') - the lab
	 * is currently open - the lab is not a feedback lab - all his requests are archived (i.e. no
	 * pending/processing requests) - the lab allows all students or the student is in the right mentor group
	 */
	public boolean canEnqueueSelfForLab(Object principal, Long id) {
		User user = getUser(principal);

		Lab lab = getLab(id);

		Course course = lab.getCourse();
		if (!user.participates(course)) {
			return false;
		}
		if (lab.isEnqueueClosed()) {
			return false;
		}
		if (lab.isBeingProcessed(user)) {
			return false;
		}
		if (!lab.isOpenOrSlotSelection()) {
			return false;
		}
		if (!lab.allAllowed()) {
			if (!lab.userAllowedForThisLab(user)) {
				return false;
			}
		}
		if (lab.isFeedbackLab()) {
			return false;
		}

		return true;
	}

	/**
	 * Checks whether a user can enqueue others for a lab.
	 *
	 * @param  principal The user who wants to enqueue others in a lab.
	 * @param  id        The lab id.
	 * @return           True if the user can enqueue others in a lab.
	 */
	public boolean canEnqueueOthersForLab(Object principal, Long id) {
		User user = getUser(principal);
		Lab lab = getLab(id);
		Course course = lab.getCourse();
		return user.isAdmin() || user.manages(course) || user.teaches(course) || user.assists(course);
	}

	/**
	 * Checks whether a user can unenroll themselves from a lab.
	 *
	 * @param  principal The user.
	 * @param  id        The lab ID.
	 * @return           True if the user can unenroll themselves.
	 */
	public boolean canUnenqueueSelfFromLab(Object principal, Long id) {
		User user = getUser(principal);
		Lab lab = getLab(id);
		Course course = lab.getCourse();

		// Only stop people from unenqueueing when they are in an exam lab.
		if (lab.isExamLab() && lab.isEnqueueClosed()) {
			return false;
		}

		return user.participates(course);
	}

	/**
	 * Checks whether a user can unenroll others from a lab.
	 *
	 * @param  principal The user.
	 * @param  id        The lab ID.
	 * @return           True if the user can unenroll others.
	 */
	public boolean canUnenqueuOthersFromLab(Object principal, Long id) {
		User user = getUser(principal);

		Lab lab = getLab(id);

		Course course = lab.getCourse();

		return user.manages(course) || user.teaches(course) || user.assists(course) || user.isAdmin();
	}

	/**
	 * A user can get the next request for the given lab if he assists or teaches the course.
	 *
	 * @param  principal
	 * @param  id
	 * @return
	 */
	public boolean canNextRequest(Object principal, Long id) {
		User user = getUser(principal);

		Lab lab = getLab(id);

		Course course = lab.getCourse();

		return user.assists(course) ||
				user.manages(course) ||
				user.teaches(course);
	}

	/**
	 * A user can access next request page if he is an assistant, manager, or teacher in *any* course.
	 *
	 * @param  principal
	 * @return
	 */
	public boolean canNextRequest(Object principal) {
		User user = getUser(principal);

		return user.isAdmin() ||
				user.getRoles().stream()
						.anyMatch(
								r -> r instanceof Assistant || r instanceof Manager || r instanceof Teacher);
	}

	public boolean canViewRequests(Object principal, Long id) {
		User user = getUser(principal);

		Course course = courseRepository.findByIdOrThrow(id);

		return user.isAdmin() ||
				user.assists(course) ||
				user.manages(course) ||
				user.teaches(course);
	}

	/**
	 * A user can view the requests page if it is an admin (default role), teacher (contextual role) or
	 * assistant (contextual role).
	 *
	 * @param  principal
	 * @return
	 */
	public boolean canViewRequests(Object principal) {
		User user = getUser(principal);

		return user.isAdmin() ||
				user.getRoles().stream()
						.anyMatch(
								r -> r instanceof Assistant || r instanceof Manager || r instanceof Teacher);
	}

	public boolean canCreateLab(Object principal, Long id) {
		User user = getUser(principal);

		Course course = courseRepository.findByIdOrThrow(id);

		return user.isAdmin() ||
				user.manages(course) ||
				user.teaches(course);
	}

	public boolean canViewRoomLayout(Object principal, Long id) {
		User user = getUser(principal);

		return user.isUser();
	}

	public boolean canEditLab(Object principal, Long id) {
		User user = getUser(principal);

		Lab lab = getLab(id);

		Course course = lab.getCourse();

		return user.isAdmin() ||
				user.manages(course) ||
				user.teaches(course);
	}

	public boolean canRemoveLab(Object principal, Long id) {
		User user = getUser(principal);

		Lab lab = getLab(id);

		Course course = lab.getCourse();

		return user.isAdmin() ||
				user.manages(course) ||
				user.teaches(course);
	}

	/**
	 * Shows whether a user can enroll/unenroll students from lab slots.
	 *
	 * @param  principal The user whose permissions are to be determined.
	 * @param  id        The id of the lab.
	 * @return           True if the user has permission to manage the lab slots.
	 */
	public boolean canManageLabSlots(Object principal, Long id) {
		User user = getUser(principal);

		Lab lab = getLab(id);

		Course course = lab.getCourse();

		return user.isAdmin() ||
				user.manages(course) ||
				user.teaches(course);
	}

	public boolean canViewNotification(Object principal, Long id) {
		User user = getUser(principal);

		Notification notification = getNotification(id);

		return notification.getUser().equals(user);
	}

	/**
	 * A user can view a request if: - the request is in a course that the user assists - the request is in a
	 * course that the user teaches - it's his/her own request
	 */
	public boolean canViewRequest(Object principal, Long id) {
		User user = getUser(principal);

		Request request = getRequest(id);

		Course course = request.getLab().getCourse();

		return request.getRequestEntity().equals(user) ||
				request.getRequestEntity().equals(request.getLab().getEntityFor(user)) ||
				user.assists(course) ||
				user.manages(course) ||
				user.teaches(course) ||
				user.isAdmin();
	}

	/**
	 * User may view feedback if user is teacher or the feedback is about the user itself.
	 */
	public boolean canViewFeedback(Object principal, Long id) {
		User user = getUser(principal);

		return user.isAdmin() || user.isTeacher() || user.getId().equals(id);
	}

	public boolean canViewDeanonimizedFeedback(Object principal, Long id) {
		User user = getUser(principal);

		return user.isAdmin() || user.isTeacher();
	}

	/**
	 * Method to check if a user is allowed to check if they are allowed to see their own feedback. They are
	 * only allowed to see if they have been a TA before or if they are a teacher or admin.
	 *
	 * @param  principal The user object.
	 * @return           Boolean value to, true if they are allowed, otherwise false.
	 */
	public boolean canViewOwnFeedback(Object principal) {
		User user = getUser(principal);
		List<Role> roleList = user.getRoles();
		return roleList.stream()
				.anyMatch(e -> e instanceof Teacher || e instanceof Manager || e instanceof Assistant)
				|| user.isAdmin();
	}

	/**
	 * A user can view a request reason if: - the request is in a course that the user assists - the request
	 * is in a course that the user teaches
	 */
	public boolean canViewRequestReason(Object principal, Long id) {
		User user = getUser(principal);

		Request request = getRequest(id);

		Course course = request.getLab().getCourse();

		return user.assists(course) ||
				user.manages(course) ||
				user.teaches(course) ||
				user.isAdmin();
	}

	/**
	 * A user can finish (i.e. approve/reject/forward) a request if: - the user is the assigned assistant for
	 * the request - the user manages the course and the request is already handled - the user teaches the
	 * course - the user is actually admin
	 */
	public boolean canFinishRequest(Object principal, Long id) {
		User user = getUser(principal);

		Request request = getRequest(id);

		Course course = request.getLab().getCourse();

		boolean isAllowedAssistant = request.getAssistant() != null && request.getAssistant().equals(user)
				&& !request.isHandled();
		boolean isAllowedManager = user.manages(course) && request.isHandled();

		return isAllowedAssistant ||
				isAllowedManager ||
				user.teaches(course) ||
				user.isAdmin();
	}

	protected User getUser(Object principal) {
		if (!(principal instanceof UserPrincipal)) {
			throw new EntityNotFoundException("User was not found");
		}

		UserPrincipal userPrincipal = (UserPrincipal) principal;

		return userRepository.findById(userPrincipal.getUser().getId()).orElseThrow();
	}

	protected Lab getLab(Long id) {
		return labRepository.findById(id)
				.orElseThrow(() -> new EntityNotFoundException("Lab was not found: " + id));
	}

	protected Role getRole(Long id) {
		return roleRepository.findById(id)
				.orElseThrow(() -> new EntityNotFoundException("Role was not found"));
	}

	protected Notification getNotification(Long id) {
		return notificationRepository.findById(id)
				.orElseThrow(() -> new EntityNotFoundException("Notification was not found: " + id));
	}

	protected Request getRequest(Long id) {
		return requestRepository.findById(id)
				.orElseThrow(() -> new EntityNotFoundException("Request was not found: " + id));
	}
}
