/*
 * Queue - A Queueing system that can be used to handle labs in higher education
 * Copyright (C) 2016-2020  Delft University of Technology
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <https://www.gnu.org/licenses/>.
 */
package nl.tudelft.ewi.queue.service;

import java.time.LocalDateTime;
import java.util.*;
import java.util.stream.Collectors;

import nl.tudelft.ewi.queue.model.*;
import nl.tudelft.ewi.queue.repository.NotificationRepository;
import nl.tudelft.ewi.queue.repository.RequestRepository;

import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.util.MultiValueMap;
import org.springframework.web.servlet.mvc.support.RedirectAttributes;

import com.google.common.collect.Lists;
import com.querydsl.core.types.Predicate;
import com.querydsl.core.types.dsl.BooleanExpression;

@Service
public class RequestService {

	private static final String netIdConstant = "{netid}";

	@Autowired
	RequestRepository requestRepository;

	@Autowired
	private NotificationRepository notificationRepository;

	@Autowired
	private NotificationService notificationService;

	/**
	 * Get the next request for this assistant/teacher from a lab.
	 *
	 * @param  assistant
	 * @return
	 */
	public synchronized Optional<Request> next(User assistant, Lab currentLab, Predicate tableFilters) {
		// first check if the assistant already has a request that is processing
		// and should be handled first
		Optional<Request> processingReq = getProcessingRequest(assistant, currentLab);
		// next, check if this assistant has a request that is already assigned to them
		Optional<Request> assignedReq = getAssignedRequest(assistant, currentLab);

		Request request;
		if (processingReq.isPresent()) {
			request = processingReq.get();
		} else if (assignedReq.isPresent()) {
			request = assignedReq.get();
		} else if (currentLab.isExamLab()) {
			request = findNextPickedRequestFromQueue(currentLab, assistant, tableFilters);
		} else {
			// current assistant has neither a request already processing or assigned
			// so find in the pending queue something suitable
			request = findNextRequestFromQueue(currentLab, assistant, tableFilters);
		}

		if (request != null) {
			assignRequest(request);
			updateQueue(request, assistant);
			return Optional.of(request);
		} else {
			return Optional.empty();
		}
	}

	/**
	 * Finds the next person which has the status {@Link Picked}.
	 *
	 * @param  currentLab   The lab for which this request is submitted
	 * @param  assistant    The assistant handeling the request
	 * @param  tableFilters The filter this assistant has enabled.
	 * @return              A request which had status picked.
	 */
	private Request findNextPickedRequestFromQueue(Lab currentLab, User assistant, Predicate tableFilters) {
		Request request = null;
		List<Request> pendingRequests = getPickedRequestsForLabWithIntervals(currentLab, assistant,
				tableFilters);

		if (!pendingRequests.isEmpty()) {
			request = pendingRequests.get(0);
			request.addEvent(new RequestAssignedEvent(request, LocalDateTime.now(), assistant));
		}
		// Reject the other students in this time slot which did not get picked.
		Request finalRequest = request;
		if (null != finalRequest) {
			currentLab.getRequests().stream().filter(Request::isPending)
					.filter(request1 -> request1.getSlot().getOpensAt()
							.equals(finalRequest.getSlot().getOpensAt()))
					.forEach(request1 -> reject(assistant, request1, "Not picked", "Not picked"));
		}
		return request;
	}

	private Request findNextRequestFromQueue(Lab currentLab, User assistant, Predicate tableFilters) {
		Request request = null;
		List<Request> pendingRequests;
		if (currentLab.isSignOffIntervals()) {
			pendingRequests = getPendingRequestsForLabWithIntervals(currentLab, assistant, tableFilters);
		} else {
			pendingRequests = getPendingRequests(currentLab, assistant, tableFilters);
		}
		if (!pendingRequests.isEmpty()) {
			request = pendingRequests.get(0);
			request.addEvent(new RequestAssignedEvent(request, LocalDateTime.now(), assistant));
		}
		return request;
	}

	private Optional<Request> getProcessingRequest(User assistant, Lab currentLab) {
		return currentLab.getProcessing().stream()
				.filter(request -> request.getAssistant() != null &&
						request.getAssistant().getId().equals(assistant.getId()))
				.findFirst();
	}

	private Optional<Request> getAssignedRequest(User assistant, Lab currentLab) {
		return currentLab.getQueue().stream()
				.filter(Request::isAssigned)
				.filter(request -> request.getAssistant() != null &&
						request.getAssistant().getId().equals(assistant.getId()))
				.findFirst();
	}

	private List<Request> getPendingRequests(Lab currentLab, User assistant, Predicate tableFilters) {
		QRequest qRequest = QRequest.request;

		Iterator<Request> pendingReq = requestRepository
				.findAll(getDefaultQueuePredicates(qRequest, currentLab, assistant, tableFilters))
				.iterator();

		return Lists.newArrayList(pendingReq);
	}

	private List<Request> getPickedRequestsForLabWithIntervals(Lab currentLab, User assistant,
			Predicate tableFilters) {
		QRequest qRequest = QRequest.request;
		LocalDateTime interval = currentLab.getAcceptInterval();
		BooleanExpression allRequestsThatAreAlreadyOpenOrOpenWithinTenMinutes = qRequest.slot.opensAt
				.before(interval);
		BooleanExpression requestForCurrentLab = qRequest.lab.id.eq(currentLab.getId());
		// Check if the assigned in the history was already assigned to this assistant
		// If so, they forwarded it to any and they do not want it anymore
		BooleanExpression requestNotAlreadyAssignedTo = qRequest.assistant.isNull()
				.or(qRequest.assistant.id.ne(assistant.getId()));
		BooleanExpression pendingRequests = qRequest.status.eq(Request.Status.PICKED);
		BooleanExpression findPickedRequests = requestForCurrentLab.and(pendingRequests).and(tableFilters)
				.and(requestNotAlreadyAssignedTo);
		Iterator<Request> pendingReq = requestRepository
				.findAll(findPickedRequests
						.and(allRequestsThatAreAlreadyOpenOrOpenWithinTenMinutes))
				.iterator();
		List<Request> requestList = Lists.newArrayList(pendingReq);
		Collections.sort(requestList);
		return requestList;
	}

	private List<Request> getPendingRequestsForLabWithIntervals(Lab currentLab, User assistant,
			Predicate tableFilters) {
		QRequest qRequest = QRequest.request;
		LocalDateTime interval = currentLab.getAcceptInterval();
		BooleanExpression allRequestsThatAreAlreadyOpenOrOpenWithinTenMinutes = qRequest.slot.opensAt
				.before(interval);
		Iterator<Request> pendingReq = requestRepository
				.findAll(getDefaultQueuePredicates(qRequest, currentLab, assistant, tableFilters)
						.and(allRequestsThatAreAlreadyOpenOrOpenWithinTenMinutes))
				.iterator();
		List<Request> requestList = Lists.newArrayList(pendingReq);
		Collections.sort(requestList);
		return requestList;
	}

	private BooleanExpression getDefaultQueuePredicates(QRequest qRequest, Lab currentLab, User assistant,
			Predicate tableFilters) {
		BooleanExpression requestForCurrentLab = qRequest.lab.id.eq(currentLab.getId());
		// Check if the assigned in the history was already assigned to this assistant
		// If so, they forwarded it to any and they do not want it anymore
		BooleanExpression requestNotAlreadyAssignedTo = qRequest.assistant.isNull()
				.or(qRequest.assistant.id.ne(assistant.getId()));
		BooleanExpression pendingRequests = qRequest.status.eq(Request.Status.PENDING)
				.or(qRequest.status.eq(Request.Status.PICKED));
		List<Assignment> reqAssignments = currentLab.getAssignments().stream()
				.filter(assignment -> assignment.allowedToHandleAssignment(assistant))
				.collect(Collectors.toList());
		BooleanExpression allowedAssignment = qRequest.assignment.in(reqAssignments);
		return requestForCurrentLab.and(pendingRequests).and(tableFilters).and(requestNotAlreadyAssignedTo)
				.and(allowedAssignment);
	}

	private void assignRequest(Request request) {
		request.addEvent(new RequestProcessingEvent(request, LocalDateTime.now()));
		requestRepository.save(request);
		notificationService.sendRequestTableUpdateForAllAssistants(request);
	}

	/**
	 * Update the lab's information for all other participants after a request is taken by a TA.
	 *
	 * @param request   the request is now being processed
	 * @param assistant the assistant who is processing it
	 */
	private void updateQueue(Request request, User assistant) {
		RequestEntity entity = request.getRequestEntity();
		if (entity instanceof User) {
			User student = (User) entity;

			notifyStudent(student, assistant, request);
			updateRemainingPositions(request.getLab(), student);
		}
	}

	/**
	 * The method used to notify a student that a TA is on their way.
	 *
	 * @param student   the student to notify
	 * @param assistant the TA who's on their way
	 * @param request   the request which is being processed
	 */
	private void notifyStudent(User student, User assistant, Request request) {
		Notification notification = createNotification(request.getLab(), assistant, student);

		notificationRepository.save(notification);
		notificationService.sendPushNotification(student, notification);
		notificationService.sendRequestProcessingUpdate(student);
	}

	/**
	 * Sends a websockets message to the students' page to update their position in the queue.
	 *
	 * @param lab    the lab which they are being updated for
	 * @param entity the user who it's being updated for
	 */
	private void updateRemainingPositions(Lab lab, RequestEntity entity) {
		List<User> users = lab.getRequests()
				.stream()
				.filter(
						r -> r.isQueued()
								&& r.getRequestEntity().isUser()
								&& r.getRequestEntity() != entity)
				.map(r -> (User) r.getRequestEntity())
				.collect(Collectors.toList());

		for (User user : users) {
			notificationService.sendQueuePositionUpdate(user, lab);
		}
	}

	/**
	 * Create notification depending on the lab's direction
	 *
	 * @param lab       the lab for which the notification is being sent
	 * @param assistant the assistant who is processing the request
	 * @param student   the student who's request it was
	 */
	public static Notification createNotification(Lab lab, User assistant, User student) {
		switch (lab.getCommunicationMethod()) {
			case STUDENT_VISIT_TA:
				return new Notification(student, lab.getCourse(), "Please visit TA",
						"Please visit assistant '" + assistant.getDisplayName() + "'.", 100);
			case TA_VISIT_STUDENT:
				return new Notification(student, lab.getCourse(), "Assistant incoming",
						"Assistant '" + assistant.getDisplayName() + "' is on their way!", 100);
			case JITSI_MEET:
				return new Notification(student, lab.getCourse(), "Jitsi call started",
						"Please join the Jitsi room with " + assistant.getDisplayName() + "!", 100);
		}

		throw new IllegalStateException(
				"Cannot create a notification for a lab with direction " + lab.getCommunicationMethod());
	}

	/**
	 * Revoke a request
	 *
	 * @param request
	 */
	public void revoke(Request request) {
		if (!request.isQueued()) {
			throw new IllegalStateException("Unable to revoke a request that is not in pending state.");
		}

		request.addEvent(new RequestRevokedEvent(request, LocalDateTime.now()));
		requestRepository.save(request);
		notificationService.sendRequestTableUpdateForAllAssistants(request);

		// move people up in the queue
		updateRemainingPositions(request.getLab(), request.getRequestEntity());
	}

	/**
	 * Remove request from lab's queue and assign the assistant.
	 *
	 * If the request was not yet taken by anybody, add request processing event as well.
	 *
	 * @param assistant The assistant that approved the request
	 * @param request   The request which will be approved and assigned a processing event if not yet present.
	 */
	public void approve(User assistant, Request request, String comment) {
		if (request.getEvents().stream().noneMatch(event -> event instanceof RequestProcessingEvent)) {
			request.addEvent(new RequestProcessingEvent(request, LocalDateTime.now()));
		}
		request.addEvent(new RequestApprovedEvent(request, assistant, LocalDateTime.now(), comment));
		requestRepository.save(request);

		Map<User, Notification> notifications = notificationService
				.createFeedbackNotification(request.getRequestEntity(), request);
		notificationRepository.saveAll(notifications.values());
		notifications.forEach((user, notification) -> notificationService
				.sendPushNotification((User) request.getRequestEntity(), notification));
		notificationService.sendRequestTableUpdateForAllAssistants(request);
	}

	/**
	 * Remove request from lab's queue and assign the assistant.
	 *
	 * @param assistant
	 * @param request
	 */
	public void reject(User assistant, Request request, String comment, String commentForStudent) {
		request.addEvent(new RequestRejectedEvent(request, assistant, LocalDateTime.now(), comment,
				commentForStudent));

		requestRepository.save(request);
		notificationService.sendRequestTableUpdateForAllAssistants(request);
	}

	/**
	 * Forward the request to another assistant.
	 *
	 * @param request
	 * @param assistant
	 */
	public void forward(Request request, User assistant, String comment) {
		if (!request.getLab().getCourse().getPrivileged().contains(assistant)) {
			throw new IllegalArgumentException("Can only forward to an assistant for this course.");
		}

		if (request.getAssistant() != null && request.getAssistant().equals(assistant)) {
			throw new IllegalArgumentException("You cannot forward a request to yourself.");
		}

		// Forward the request, add forwarded event and assign to new TA.
		request.addEvent(new RequestForwardedEvent(request, LocalDateTime.now(), assistant, comment));
		request.addEvent(new RequestAssignedEvent(request, LocalDateTime.now(), assistant));

		requestRepository.save(request);
		notificationService.sendRequestTableUpdateForAllAssistants(request);
	}

	public void forwardToAny(Request request, String reason) {
		request.addEvent(new RequestForwardToAnyEvent(request, LocalDateTime.now(), reason));
		requestRepository.save(request);
	}

	public void notFound(Request request, User assistant) {
		request.addEvent(new StudentNotFoundEvent(request, LocalDateTime.now(), assistant));
		requestRepository.save(request);
		Map<User, Notification> notifications = notificationService
				.createNotFoundNotification(request.getRequestEntity(), request);
		notificationRepository.saveAll(notifications.values());
		notifications.forEach((user, notification) -> notificationService
				.sendPushNotification((User) request.getRequestEntity(), notification));
		notificationService.sendRequestTableUpdateForAllAssistants(request);
	}

	public String setSubmissionUrl(Request request) {
		RequestEntity requestEntity = request.getRequestEntity();
		if (requestEntity instanceof User) {
			User student = (User) requestEntity;
			String templateSubmissionUrl = request.getLab().getCourse().getSubmissionUrl();
			if (templateSubmissionUrl != null && !templateSubmissionUrl.isEmpty()) {
				templateSubmissionUrl = templateSubmissionUrl.replace(netIdConstant, student.getUsername());
				return templateSubmissionUrl;
			}
		}
		return "";
	}

	public String redirectToRequestWithParams(RedirectAttributes redirectAttributes,
			MultiValueMap<String, String> parameters) {
		// redirect to apply these params to the predicate so the table is actually filtered
		updateRedirectAttributesNoSquash(redirectAttributes, parameters);
		return "redirect:/requests";
	}

	public void updateRedirectAttributes(RedirectAttributes redirectAttributes,
			MultiValueMap<String, String> parameters) {
		for (Map.Entry<String, List<String>> entry : parameters.entrySet()) {
			for (String value : entry.getValue()) {
				redirectAttributes.addAttribute(entry.getKey(), value);
			}
		}
	}

	/**
	 * Update the redirect attributes without squashing lists into a string. NOTE: This method relies on a
	 * flaw in the supporting library, and as such may well break in the future.
	 *
	 * @param redirectAttributes The attributes to be updated.
	 * @param parameters         The values with which to update the attributes.
	 */
	@SuppressWarnings("unchecked")
	public void updateRedirectAttributesNoSquash(RedirectAttributes redirectAttributes,
			MultiValueMap<String, String> parameters) {
		for (Map.Entry<String, List<String>> entry : parameters.entrySet()) {

			//noinspection unchecked
			((HashMap<String, Object>) redirectAttributes).putIfAbsent(entry.getKey(),
					entry.getValue());
		}
	}
}
