/*
 * Queue - A Queueing system that can be used to handle labs in higher education
 * Copyright (C) 2016-2020  Delft University of Technology
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <https://www.gnu.org/licenses/>.
 */
package nl.tudelft.ewi.queue.service;

import java.util.*;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;

import javax.servlet.http.HttpServletRequest;

import nl.tudelft.ewi.queue.model.*;
import nl.tudelft.ewi.queue.repository.RequestRepository;
import nl.tudelft.ewi.queue.repository.RequestTypeRepository;

import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.domain.Page;
import org.springframework.data.domain.Pageable;
import org.springframework.stereotype.Service;
import org.springframework.ui.Model;
import org.springframework.util.LinkedMultiValueMap;
import org.springframework.util.MultiValueMap;

import com.google.common.collect.Lists;
import com.google.common.collect.Sets;
import com.querydsl.core.types.Predicate;
import com.querydsl.core.types.dsl.BooleanExpression;

/**
 * A class to manage the storing and retrieving of information for the request table view.
 *
 * Filters are being stored in the session in two parts: the list of parameters that were provided upon
 * filtering and the predicate that was generated from those parameters. The session for one user will contain
 * this information for every page the user has submitted a filter on until their session expires. Filters are
 * being stored this way to make request table pages less dependent on the parameter list being passed
 * correctly. Before doing so, having the parameter list right when entering a /requests page was crucial to
 * getting the right filters.
 *
 * Filters are stored under page-specific keys. The key format is as follows:
 * `[page]_filter_[predicate|parameters]` in which `page` is the path of the current request follows,
 * `predicate` is used for storing predicates and `parameters` is used for storing parameters.
 */
@Service
public class RequestTableService {
	private static final String PREDICATE_POSTFIX = "_filter_predicate";
	private static final String PARAMETERS_POSTFIX = "_filter_parameters";
	private static final Set<String> FILTERS = Sets.newHashSet(
			"lab.course", "lab", "assignment", "room", "assistant", "status", "requestType");

	@Autowired
	private RequestTypeRepository requestTypeRepository;

	@Autowired
	private RequestRepository requestRepository;

	/**
	 * Analyses the given parameters and performs an action based on it. If the parameters contain a 'clear'
	 * attribute, the filters stored in the current session are removed. If the parameters do not contain a
	 * filtering attribute, nothing is submitted. Otherwise, the given parameters and predicate are stored in
	 * the current session.
	 *
	 * @param request    The HttpServletRequest used to get the session.
	 * @param parameters The parameters to store.
	 * @param predicate  The predicate generated by the parameters to store.
	 */
	public void submitFilters(HttpServletRequest request,
			MultiValueMap<String, String> parameters,
			Predicate predicate) {

		if (containsFilterParams(parameters)) {
			String key = pathFromRequest(request);
			if (parameters.containsKey("clear")) {
				clearFilters(request, key);
			} else {
				storeFilters(request, key, parameters, predicate);
			}
		}
	}

	/**
	 * Stores the given parameters and predicates under the given key in the current session.
	 *
	 * @param request    The HttpServletRequest used to get the session.
	 * @param key        The key to store values under.
	 * @param parameters The parameters to store.
	 * @param predicate  The predicate generated by the parameters to store.
	 */
	private void storeFilters(HttpServletRequest request,
			String key,
			MultiValueMap<String, String> parameters,
			Predicate predicate) {
		request.getSession().setAttribute(key + PARAMETERS_POSTFIX, parameters);
		request.getSession().setAttribute(key + PREDICATE_POSTFIX, predicate);
	}

	/**
	 * Clears the filter currently stored in the session under the given key.
	 *
	 * @param request The HttpServletRequest used to get the session.
	 * @param key     The key to find and clear stored values under.
	 */
	private void clearFilters(HttpServletRequest request,
			String key) {
		request.getSession().removeAttribute(key + PARAMETERS_POSTFIX);
		request.getSession().removeAttribute(key + PREDICATE_POSTFIX);
	}

	/**
	 * Fills the model of a table of current requests appropriately. This method uses the request parameter to
	 * determine a key under which a filter may be previously stored and uses that filter if possible to
	 * filter out requests. This method also filters the list of requests to match only the requests that
	 * should be answered by the given user excluding the requests they have intentionally filtered out
	 * through the stored Predicate.
	 *
	 * @param request  The HttpServletRequest used to get the session.
	 * @param user     The user requesting their current request table.
	 * @param pageable A Pageable object governing pagination of requests.
	 * @param model    The Model to fill in.
	 */
	public void fillCurrentTableModel(HttpServletRequest request,
			User user,
			Pageable pageable,
			Model model) {
		String key = pathFromRequest(request);

		// Get existing filter information if it exists.
		Object filters = getFiltersFromSession(request, key);
		Predicate predicate = getPredicateFromSession(request, key);

		// Find the currently active labs and courses the given user should be
		// able to see in the queue.
		List<Course> courses = Stream
				.of(user.getTeaches().stream(), user.getManages().stream(), user.getAssists().stream())
				.flatMap(Function.identity())
				.filter(course -> !course.getIsArchived())
				.collect(Collectors.toList());

		List<Lab> activeLabs = activeLabs(courses);
		List<Long> activeLabIds = activeLabs.stream()
				.map(Lab::getId)
				.collect(Collectors.toList());

		// Filter the requests on which the user is allowed to handle.
		BooleanExpression allowedToHandle = QRequest.request.assignment.assistants.contains(user)
				.or(QRequest.request.assignment.assistants.isEmpty());
		// Filter the requests on which is in an active lab.
		BooleanExpression filterOnActiveLabs = QRequest.request.lab.id.in(activeLabIds);
		Page<Request> filteredRequests = requestRepository.findAll(
				filterOnActiveLabs.and(predicate).and(allowedToHandle), pageable);

		// Count the number of requests there are per course.
		Map<Long, Long> queueCount = activeLabs.stream()
				.collect(Collectors.toMap(Lab::getId,
						lab -> {
							List<Request> requests = Lists.newArrayList(requestRepository.findAll(
									QRequest.request.lab.id.eq(lab.getId()).and(predicate)
											.and(allowedToHandle)));
							return lab.getQueuedCount(requests, user);
						}));

		// Add the created model attributes
		addToModel(model, courses, activeLabs, filteredRequests, filters);
		model.addAttribute("queued", queueCount);
	}

	/**
	 * Fills the model of a table of historical requests appropriately. This method uses the request parameter
	 * to determine a key under which a filter may be previously stored and uses that filter if possible to
	 * filter out requests. This method also filters the list of requests to match only the requests that were
	 * once posted by the given user, excluding the requests they have intentionally filtered out through the
	 * stored Predicate and the ones not matching with the corresponding course ID.
	 *
	 * @param request  The HttpServletRequest used to get the session.
	 * @param user     The user requesting their history request table.
	 * @param pageable A Pageable object governing pagination of requests.
	 * @param model    The Model to fill in.
	 * @param courseId The course ID to filter the requests on.
	 */
	public void fillHistoricTableModel(HttpServletRequest request,
			User user,
			Pageable pageable,
			Model model,
			Long courseId) {
		String key = pathFromRequest(request);

		// Get existing filter information if it exists.
		Object filters = getFiltersFromSession(request, key);
		Predicate predicate = getPredicateFromSession(request, key);

		// Find all labs and courses the given user should be able to see in
		// the queue.
		List<Course> courses = user.getParticipates();
		List<Lab> labs = labs(courses);

		List<Group> groups = courses.stream()
				.filter(Course::getHasGroups)
				.flatMap(c -> c.getGroups().stream())
				.filter(g -> g.hasMember(user))
				.collect(Collectors.toList());

		// Filter the requests on which is created by this user or its group.
		QRequest qRequest = QRequest.request;
		BooleanExpression currentUser = qRequest.requestEntity.id.eq(user.getId());
		BooleanExpression currentGroup = qRequest.requestEntity.in(groups);

		// If a specific course is selected, only show requests for that course.
		if (courseId != -1L) {
			predicate = qRequest.lab.course.id.eq(courseId).and(predicate);
		}

		// Find all the requests from the user or the associated group
		Page<Request> filteredRequests = requestRepository.findAll(
				(currentGroup.or(currentUser)).and(predicate), pageable);

		// Add the created model attributes.
		addToModel(model, courses, labs, filteredRequests, filters);
	}

	/**
	 * Adds the given information to the model using the variable names used in the "request/list.html" page.
	 *
	 * @param model    The model to The model to add attributes to.
	 * @param courses  The courses that should be displayed in filters.
	 * @param labs     The labs that should be displayed in filters.
	 * @param requests The requests that should be displayed on the page.
	 * @param filters  The state of the filters.
	 */
	private void addToModel(Model model,
			List<Course> courses,
			List<Lab> labs,
			Page<Request> requests,
			Object filters) {
		model
				.addAttribute("rooms", rooms(labs))
				.addAttribute("requestTypes", requestTypeRepository.findAll())
				.addAttribute("courses", courses)
				.addAttribute("activeLabs", labs)
				.addAttribute("requests", requests)
				.addAttribute("state", filters)
				.addAttribute("assignments", assignments(labs))
				.addAttribute("assistants", assistants(courses));
	}

	/**
	 * @param  request The HttpServletRequest used to get the session.
	 * @param  key     The key to find the predicate with.
	 * @return         The predicate stored under the given key in the session.
	 */
	public Predicate getPredicateFromSession(HttpServletRequest request,
			String key) {
		return (Predicate) request.getSession().getAttribute(key + PREDICATE_POSTFIX);
	}

	/**
	 * @param  request The HttpServletRequest used to get the session.
	 * @param  key     The key to find the parameters with.
	 * @return         The parameters stored under the given key in the session.
	 */
	private Object getFiltersFromSession(HttpServletRequest request,
			String key) {
		return Optional
				.ofNullable(request.getSession().getAttribute(key + PARAMETERS_POSTFIX))
				.orElse(new LinkedMultiValueMap<>());
	}

	/**
	 * Checks whether the query parameters include filters.
	 *
	 * @param  parameters the query parameters of the request
	 * @return            true if there is at least one filtering query parameter
	 */
	private boolean containsFilterParams(MultiValueMap<String, String> parameters) {
		return parameters != null && !(Sets.intersection(FILTERS, parameters.keySet())).isEmpty();
	}

	/**
	 * Flatmaps a list of courses to a list of labs.
	 *
	 * @param  courses The courses to find labs from.
	 * @return         The total list of labs from the given courses.
	 */
	private List<Lab> labs(List<Course> courses) {
		return courses.stream()
				.flatMap(course -> course.getLabs().stream())
				.collect(Collectors.toList());
	}

	/**
	 * Flatmaps a list of courses to a list of active labs.
	 *
	 * @param  courses The courses to find labs from.
	 * @return         The total list of active labs from the given courses.
	 */
	private List<Lab> activeLabs(List<Course> courses) {
		return labs(courses).stream()
				.filter(Lab::display)
				.collect(Collectors.toList());
	}

	/**
	 * Flatmaps the given labs to a list of distinct assignments.
	 *
	 * @param  labs The labs to find assignments from.
	 * @return      The total list of assignments from the given labs.
	 */
	private List<Assignment> assignments(List<Lab> labs) {
		return labs.stream()
				.flatMap(lab -> lab.getAssignments().stream())
				.distinct()
				.collect(Collectors.toList());
	}

	/**
	 * Flatmaps the given labs to a list of distinct rooms.
	 *
	 * @param  labs The labs in which to find rooms.
	 * @return      The list of unique rooms available to the given labs.
	 */
	private List<Room> rooms(List<Lab> labs) {
		return labs.stream()
				.flatMap(lab -> lab.getRooms().stream())
				.distinct()
				.collect(Collectors.toList());
	}

	/**
	 * Flatmaps the given courses to a list of distinct request-assignable users.
	 *
	 * @param  courses The courses to find assistants in.
	 * @return         The list of unique assistants in the given set of courses.
	 */
	private List<User> assistants(List<Course> courses) {
		return courses.stream()
				.flatMap(course -> course.getPrivileged().stream())
				.distinct()
				.sorted(Comparator.comparing(RequestEntity::getDisplayName))
				.collect(Collectors.toList());
	}

	/**
	 * Uses the given HttpServletRequest to extract the path of the current endpoint. This can in turn be used
	 * as the key to store filters on.
	 *
	 * @param  request The request.
	 * @return         The (normalized) key/path extracted from the request.
	 */
	private String pathFromRequest(HttpServletRequest request) {
		String key = request.getServletPath();
		if (!key.startsWith("/")) {
			key = "/" + key;
		}
		if (key.endsWith("/")) {
			key = key.substring(0, key.length() - 1);
		}
		return key;
	}
}
