/*
 * Queue - A Queueing system that can be used to handle labs in higher education
 * Copyright (C) 2016-2020  Delft University of Technology
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <https://www.gnu.org/licenses/>.
 */
package nl.tudelft.ewi.queue.service;

import java.util.List;

import nl.tudelft.ewi.queue.model.DefaultRole;
import nl.tudelft.ewi.queue.model.FirstYearStudent;
import nl.tudelft.ewi.queue.model.User;
import nl.tudelft.ewi.queue.model.UserPrincipal;
import nl.tudelft.ewi.queue.repository.FirstYearStudentRepository;
import nl.tudelft.ewi.queue.repository.UserRepository;

import org.apache.commons.lang3.StringUtils;
import org.opensaml.saml2.core.Attribute;
import org.opensaml.xml.XMLObject;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.security.core.userdetails.UsernameNotFoundException;
import org.springframework.security.saml.SAMLCredential;
import org.springframework.security.saml.userdetails.SAMLUserDetailsService;
import org.springframework.stereotype.Service;

@Service
public class SAMLUserDetailsServiceImpl implements SAMLUserDetailsService {

	private static final String DISPLAY_NAME = "urn:mace:dir:attribute-def:displayName";
	private static final String MAIL = "mail";
	private static final String EDU_PERSON_AFFILIATION = "urn:mace:dir:attribute-def:eduPersonAffiliation";
	private static final String TUD_STUDENT_NUMBER = "tudStudentNumber";

	private static final Logger logger = LoggerFactory.getLogger(SAMLUserDetailsServiceImpl.class);

	@Autowired
	private UserRepository userRepository;

	@Autowired
	private FirstYearStudentRepository firstYearStudentRepository;

	/**
	 * Load a user. If the user does not exist, it is implicitly created.
	 *
	 * @param  credential
	 * @return
	 * @throws UsernameNotFoundException
	 */
	@Override
	public Object loadUserBySAML(SAMLCredential credential) throws UsernameNotFoundException {
		String nameId = credential.getNameID().getValue();
		User user = userRepository.findByUsername(nameId);

		if (null == user) {
			logger.info("Implicitly create user " + nameId);
			user = createUser(nameId, credential);
			saveUserWithCheck(user);
		}

		logger.info(nameId + " is logged in");

		logger.info("SAML credentials: " + credential.toString());
		logger.info("SAML credentials details:\n");
		for (Attribute attribute : credential.getAttributes()) {
			logger.info("*** attribute: " + attribute.getName() + " (" + attribute.getFriendlyName() + ")");
			logger.info("****** value: " + credential.getAttributeAsString(attribute.getName()));
			for (XMLObject value : attribute.getAttributeValues()) {
				logger.debug("****** raw value:" + value.toString());
			}
		}

		String email = credential.getAttributeAsString(MAIL);
		String displayName = credential.getAttributeAsString(DISPLAY_NAME);
		if (!email.equals(user.getEmail())) {
			logger.info(String.format(
					"User with netid %s changed email from %s to %s",
					nameId, user.getEmail(), email));
			user.setEmail(email);
			saveUserWithCheck(user);
		}
		if (!displayName.equals(user.getDisplayName())) {
			logger.info(String.format(
					"User with netid %s changed display name from %s to %s",
					nameId, user.getDisplayName(), displayName));
			user.setDisplayName(displayName);
			saveUserWithCheck(user);
		}
		if (user.getDefaultRole() == DefaultRole.ROLE_STUDENT) {
			int studentNumber = Integer.parseUnsignedInt(credential.getAttributeAsString(TUD_STUDENT_NUMBER));
			if (studentNumber != user.getStudentNumber()) {
				user.setStudentNumber(studentNumber);
				saveUserWithCheck(user);
			}
		}

		AttachUserToFirstYearStudent(user);
		return new UserPrincipal(user);
	}

	/**
	 * Saves the user to the user repository and if this fails, remove all the accents in the name and try
	 * again.
	 *
	 * @param user The user which needs to be saved.
	 */
	private void saveUserWithCheck(User user) {
		try {
			userRepository.save(user);
		} catch (Exception e) {
			String newName = StringUtils.stripAccents(user.getDisplayName());
			user.setDisplayName(newName);
			userRepository.save(user);
		}
	}

	private void AttachUserToFirstYearStudent(User user) {
		List<FirstYearStudent> firstYearStudents = firstYearStudentRepository.findByNetId(user.getUsername());
		if (firstYearStudents != null && !firstYearStudents.isEmpty()) {
			for (FirstYearStudent firstYearStudent : firstYearStudents) {
				if (firstYearStudent != null && firstYearStudent.getUser() == null) {
					firstYearStudent.setUser(user);
					firstYearStudentRepository.save(firstYearStudent);
				}
			}
		}
		firstYearStudents = firstYearStudentRepository.findByNetId(String.valueOf(user.getStudentNumber()));
		if (firstYearStudents != null && !firstYearStudents.isEmpty()) {
			for (FirstYearStudent firstYearStudent : firstYearStudents) {
				if (firstYearStudent != null && firstYearStudent.getUser() == null) {
					firstYearStudent.setUser(user);
					firstYearStudentRepository.save(firstYearStudent);
				}
			}
		}
	}

	/**
	 * Create a new user.
	 *
	 * @param  username
	 * @param  credential
	 * @return            a new user
	 */
	private static User createUser(String username, SAMLCredential credential) {
		DefaultRole role = DefaultRole.ROLE_STUDENT;
		String displayName = credential.getAttributeAsString(DISPLAY_NAME);
		String email = credential.getAttributeAsString(MAIL);
		int studentNumber = -1;
		String roleAttribute = credential.getAttributeAsString(EDU_PERSON_AFFILIATION);

		if (roleAttribute == null) {
			logger.warn("SAML User signed in with null role:\n");
			logger.warn("****** username: " + displayName);
		} else {
			if (roleAttribute.equals("employee")) {
				role = DefaultRole.ROLE_TEACHER;
			} else if (roleAttribute.equals("student")) {
				studentNumber = Integer.parseUnsignedInt(credential.getAttributeAsString(TUD_STUDENT_NUMBER));
			}
		}

		return new User(username, "", displayName, email, role, studentNumber);
	}
}
