/*
 * Queue - A Queueing system that can be used to handle labs in higher education
 * Copyright (C) 2016-2020  Delft University of Technology
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <https://www.gnu.org/licenses/>.
 */
package nl.tudelft.ewi.queue.tasks;

import java.time.LocalDateTime;
import java.util.*;
import java.util.stream.Collectors;

import javax.transaction.Transactional;

import nl.tudelft.ewi.queue.model.*;
import nl.tudelft.ewi.queue.repository.LabRepository;
import nl.tudelft.ewi.queue.repository.RequestRepository;
import nl.tudelft.ewi.queue.service.LabService;

import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.scheduling.annotation.Scheduled;
import org.springframework.stereotype.Component;

import com.querydsl.core.types.dsl.BooleanExpression;

@Component
@Transactional
public class RequestTasks {

	@Autowired
	private RequestRepository requestRepository;

	@Autowired
	private LabService labService;

	@Autowired
	private LabRepository labRepository;

	/**
	 * Checks whether there are upcoming timeslots which TAs need to see in their request tables. Any
	 * timeslots that are opened some time (10 minutes) after the current time should be shown.
	 *
	 * Additionally, it is checked whether the timeslot appears to be of a lab of the current day (it opens at
	 * most 1 day before now).
	 */
	@Scheduled(cron = "0 0/5 * * * ?")
	public void checkForUpcomingRequestsSlot() {
		QRequest qRequest = QRequest.request;
		LocalDateTime now = LocalDateTime.now();

		BooleanExpression shouldShow = qRequest.slot.opensAt.between(now.minusDays(1), now.plusMinutes(10L))
				.and(qRequest.status.eq(Request.Status.PENDING)
						.or(qRequest.status.eq(Request.Status.PICKED)));

		final Iterable<Request> requests = requestRepository.findAll(shouldShow);
		for (Request request : requests) {
			labService.createAndSendNotifications(request);
		}
	}

	@Scheduled(cron = "0 0/10 * * * ?")
	public void randomSelectForExamLab() {
		QLab qLab = QLab.lab;
		BooleanExpression startsInTenMinutes = qLab.slot.opensAt.between(LocalDateTime.now(),
				LocalDateTime.now().plusMinutes(10)).and(qLab.examLab.isTrue());
		Iterable<Lab> examLabs = labRepository.findAllByExamLabIsTrue(startsInTenMinutes);
		for (Lab examLab : examLabs) {
			run(examLab);
		}
	}

	private void run(Lab lab) {
		Map<LocalDateTime, List<Request>> requests = getSlotRequestsMap(lab);
		lab.setEnqueueClosed(true);
		for (LocalDateTime localDateTime : requests.keySet()) {
			List<Request> reqList = requests.get(localDateTime).stream()
					.filter(request -> !request.getStatus().equals(Request.Status.REVOKED))
					.collect(Collectors.toList());
			Collections.shuffle(reqList);
			double studentsToPick = (double) reqList.size() * lab.getExamLabPercentage() / 100;

			for (int i = 0; i < Math.min(Math.ceil(studentsToPick), reqList.size()); i++) {
				reqList.get(i).addEvent(new RequestPickedEvent(reqList.get(i), LocalDateTime.now()));
			}
		}
	}

	private Map<LocalDateTime, List<Request>> getSlotRequestsMap(Lab lab) {
		Map<LocalDateTime, List<Request>> map = new HashMap<>();

		for (Request request : lab.getRequests()) {
			if (!map.containsKey(request.getSlot().getOpensAt())) {
				map.put(request.getSlot().getOpensAt(), new ArrayList<Request>());
			}
			map.get(request.getSlot().getOpensAt()).add(request);
		}
		return map;
	}
}
