/*
 * Queue - A Queueing system that can be used to handle labs in higher education
 * Copyright (C) 2016-2020  Delft University of Technology
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <https://www.gnu.org/licenses/>.
 */
package nl.tudelft.ewi.queue.validator;

import java.util.ArrayList;
import java.util.Collections;
import java.util.List;

import nl.tudelft.ewi.queue.model.Lab;
import nl.tudelft.ewi.queue.model.RequestType;

import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;
import org.springframework.validation.Errors;
import org.springframework.validation.Validator;

@Component
public class LabValidator implements Validator {
	@Autowired
	private SlotValidator slotValidator;

	@Override
	public boolean supports(Class<?> clazz) {
		return Lab.class.isAssignableFrom(clazz);
	}

	@Override
	public void validate(Object target, Errors errors) {
		Lab targetLab = (Lab) target;

		// Check request types for at least one valid pointer.
		errors.pushNestedPath("requesttype");
		List<RequestType> allowedRequestTypes = new ArrayList<>(targetLab.getAllowedRequestTypes());
		allowedRequestTypes.removeAll(Collections.singleton(null));
		if (allowedRequestTypes.isEmpty()) {
			errors.reject("At least one Request type must be selected");
		}
		errors.popNestedPath();

		errors.pushNestedPath("slot");
		slotValidator.validate(targetLab.getSlot(), errors);
		errors.popNestedPath();

		// Check that an exam lab is also a sign of with interval lab
		errors.pushNestedPath("examlab");
		if (targetLab.isExamLab() && !targetLab.isSignOffIntervals()) {
			errors.reject("An exam lab must have intervals");
		}
		errors.popNestedPath();
	}
}
