/*
 * Queue - A Queueing system that can be used to handle labs in higher education
 * Copyright (C) 2016-2020  Delft University of Technology
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <https://www.gnu.org/licenses/>.
 */
package nl.tudelft.ewi.queue.validator;

import nl.tudelft.ewi.queue.forms.ParticipantForm;
import nl.tudelft.ewi.queue.model.Course;
import nl.tudelft.ewi.queue.model.User;
import nl.tudelft.ewi.queue.repository.CourseRepository;
import nl.tudelft.ewi.queue.repository.UserRepository;

import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;
import org.springframework.validation.Errors;
import org.springframework.validation.Validator;

@Component
public class ParticipantValidator implements Validator {
	@Autowired
	CourseRepository courseRepository;

	@Autowired
	UserRepository userRepository;

	@Override
	public boolean supports(Class<?> clazz) {
		return ParticipantForm.class.isAssignableFrom(clazz);
	}

	@Override
	public void validate(Object target, Errors errors) {
		ParticipantForm participantForm = (ParticipantForm) target;

		String username = User.guaranteeValidNetId(participantForm.getUsername());
		User user = getUser(username);

		if (user == null) {
			errors.rejectValue("username", "user.username.notFound");
		} else {
			Course course = getCourse(participantForm.getCourseId());

			if (user.participates(course) || user.assists(course) || user.teaches(course)) {
				errors.rejectValue("username", "user.username.participates");
			}
		}
	}

	protected User getUser(String username) {
		return userRepository.findByUsername(username);
	}

	protected Course getCourse(long courseId) {
		return courseRepository.findById(courseId).orElseThrow();
	}
}
