/*
 * Queue - A Queueing system that can be used to handle labs in higher education
 * Copyright (C) 2016-2020  Delft University of Technology
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <https://www.gnu.org/licenses/>.
 */
package nl.tudelft.ewi.queue.viewmodel;

import javax.validation.constraints.NotNull;

import lombok.Data;
import nl.tudelft.ewi.queue.model.*;

/**
 * A data class that can translate the intention of creating a new Request into an actually new Request. This
 * is a selective copy of the data held by the Request Entity, but without the coupledness to the database.
 * This is independently passed along as the model which is filled in by users and finally passed along to the
 * server to be converted into a Request in the database.
 */
@Data
public class RequestCreateModel {
	private RequestSlot slot;

	@NotNull
	private Room room;

	private Boolean jitsi;

	private RequestEntity requestEntity;

	@NotNull
	private RequestType requestType;

	private String comment;
	private String question;

	private Lab lab;

	@NotNull
	private Assignment assignment;

	public RequestCreateModel() {
	}

	/**
	 * Converts this RequestCreateModel into an actual Request object.
	 *
	 * @return The Request object created from the data in this object.
	 */
	public Request convert() {
		Request r = new Request();
		r.setSlot(slot);
		r.setRoom(room);
		r.setRequestEntity(requestEntity);
		r.setRequestType(requestType);
		r.setComment(comment);
		r.setQuestion(question);
		r.setLab(lab);
		r.setAssignment(assignment);
		if (jitsi != null && jitsi) {
			r.setJitsiRoom();
		}
		return r;
	}

	/**
	 * @return A boolean representing whether the conversion from this model object can happen and is valid.
	 */
	public boolean hasErrors() {
		if (lab == null || requestEntity == null) {
			return true;
		}

		final boolean hasQuestionType = "question".equalsIgnoreCase(requestType.getName());
		final boolean hasQuestion = question != null && question.length() >= 15;

		final boolean assignmentIsIn = lab.containsAssignment(assignment);
		final boolean roomIsIn = lab.hasRoom(room);
		final boolean typeIsIn = lab.containsAllowedRequestType(requestType);

		return !assignmentIsIn || !roomIsIn || !typeIsIn || (hasQuestionType && !hasQuestion);
	}
}
