/*
 * Queue - A Queueing system that can be used to handle labs in higher education
 * Copyright (C) 2016-2020  Delft University of Technology
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <https://www.gnu.org/licenses/>.
 */
var selectedCourses;
var selectedLabs;
var selectedAssignments;
var selectedRooms;
var selectedAssigned;
var selectedStatuses;
var selectedTypes;

$(function() {
    selectedCourses = $("#courseSelect").find(":selected").text() || $("#courseSelect").find("option").text();
    selectedLabs = $("#labSelect").find(":selected").text() || $("#labSelect").find("option").text();
    selectedAssignments = $("#assignmentSelect").find(":selected").text() || $("#assignmentSelect").find("option").text();
    selectedRooms = $("#roomSelect").find(":selected").text() || $("#roomSelect").find("option").text();
    selectedAssigned = $("#assignedSelect").find(":selected").text() || $("#assignedSelect").find("option").text();
    selectedStatuses = $("#statusSelect").find(":selected").text() || $("#statusSelect").find("option").text();
    selectedTypes = $("#requestTypeSelect").find(":selected").text() || $("#requestTypeSelect").find("option").text();
});

function connect() {
    let protocol = "wss:";
    if (window.location.protocol !== "https:") {
        protocol = "ws:";
    }
    const socket = new WebSocket(protocol + '//' + window.location.host + '/stomp');

    stompClient = Stomp.over(socket);
    stompClient.connect({}, function (frame) {
        stompClient.subscribe('/user/queue/notifications', function (notification) {
            var notificationObject = JSON.parse(notification.body);

            if (notificationObject["request"] !== undefined) {
                updateRequest(notificationObject["request"]);
            }

            if (notificationObject["notification"] !== undefined) {
                showNotification(notificationObject);
            }
        });
    });
}

function updateRequest(requestInfo) {
    if (requestAlreadyInTable(requestInfo)) {
        updateExistingRequestInTable(requestInfo);
    } else if (requestInfo["display"] === 'true' && inFilter(requestInfo)) {
        prependToRequestTable(requestInfo);
        increaseGetNextCounter(requestInfo["labId"]);
    }
}

function inFilter(requestInfo) {
    return selectedCourses.includes(requestInfo["course"]) &&
        selectedLabs.includes(requestInfo["lab"]) &&
        selectedAssignments.includes(requestInfo["assignment"]) &&
        selectedRooms.includes(requestInfo["room"]) &&
        (selectedAssigned.includes($("#assignedSelect").find("option").text())
            || selectedAssigned.includes(requestInfo["assigned"])) &&
        selectedStatuses.includes(requestInfo["status"]) &&
        selectedTypes.includes(requestInfo["type"]);
}

function requestAlreadyInTable(requestInfo) {
    var rowSelector = $("#request-" + requestInfo.id);
    return rowSelector.length > 0;
}

function updateExistingRequestInTable(requestInfo) {
    var statusSelector = getStatusSelector(requestInfo);
    statusSelector.text(requestInfo.status);
    statusSelector.removeClass();
    statusSelector.addClass('badge badge-pill bg-info text-white');

    var rowSelector = getRowSelector(requestInfo);
    rowSelector.removeClass();
    rowSelector.addClass('text-white');
    rowSelector.addClass(bgColorDict[requestInfo["status"]]);

    if (requestInfo["assigned"] !== undefined) {
        var assignedSelector = getAssignedSelector(requestInfo);
        assignedSelector.text(requestInfo["assigned"]);
    }

    if (requestInfo["status"] !== 'PENDING') {
        decreaseGetNextCounter(requestInfo["labId"]);
    }
}

function prependToRequestTable(requestInfo) {
    var source   = $("#request-entry-template").html();
    var template = Handlebars.compile(source);
    var html = template(requestInfo);
    $(html).hide().prependTo('#requesttable tbody').fadeIn();
}

function decreaseGetNextCounter(labId) {
    var lab = $('#lab-' + labId);
    var span = $('#span-' + labId);

    var counter = parseInt(span.text());
    if (counter !== 0) {
        counter = counter - 1;
        span.text(counter);
        if (counter === 0) {
            lab.hide();
        }
    }

    // TODO: if counter hits 0, remove the button to get next request.
}

function increaseGetNextCounter(labId) {
    var labIdentifier = '#lab-' + labId;
    var spanIdentifier = '#span-' + labId;

    $(spanIdentifier).html(parseInt($(spanIdentifier).text()) + 1);
    $(labIdentifier).show();
}

function getStatusSelector(requestInfo) {
    return $("#status-" + requestInfo.id);
}

function getRowSelector(requestInfo) {
    return $("#request-" + requestInfo.id);
}

function getAssignedSelector(requestInfo) {
    return $("#assigned-" + requestInfo.id);
}

function disconnect() {
    if (stompClient != null) {
        stompClient.disconnect();
    }
}

bgColorDict = {'REJECTED': 'bg-rejected',
    'APPROVED': 'bg-approved',
    'PROCESSING': 'bg-processing',
    'REVOKED': 'bg-revoked',
    'FORWARDED': 'bg-forwarded',
    'PENDING': 'bg-processing',
    'ASSIGNED': 'bg-processing'};
