/*
 * Queue - A Queueing system that can be used to handle labs in higher education
 * Copyright (C) 2016-2020  Delft University of Technology
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <https://www.gnu.org/licenses/>.
 */
package nl.tudelft.ewi.queue.controller;

import static org.springframework.test.web.servlet.request.MockMvcRequestBuilders.get;
import static org.springframework.test.web.servlet.result.MockMvcResultMatchers.*;

import nl.tudelft.ewi.queue.QueueApplication;
import nl.tudelft.ewi.queue.model.Course;
import nl.tudelft.ewi.queue.repository.CourseRepository;
import nl.tudelft.ewi.queue.repository.RequestRepository;
import nl.tudelft.ewi.queue.repository.UserRepository;

import org.junit.jupiter.api.BeforeEach;
import org.junit.jupiter.api.Test;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.boot.test.autoconfigure.web.servlet.AutoConfigureMockMvc;
import org.springframework.boot.test.context.SpringBootTest;
import org.springframework.security.test.context.support.WithUserDetails;
import org.springframework.test.annotation.DirtiesContext;
import org.springframework.test.web.servlet.MockMvc;

import com.fasterxml.jackson.databind.ObjectMapper;

@SpringBootTest(classes = QueueApplication.class)
@AutoConfigureMockMvc
@DirtiesContext(classMode = DirtiesContext.ClassMode.BEFORE_CLASS)
public class CourseControllerTest {

	@Autowired
	RequestRepository requestRepository;

	@Autowired
	CourseRepository courseRepository;

	@Autowired
	UserRepository userRepository;

	@Autowired
	ObjectMapper objectMapper;

	@Autowired
	private MockMvc mockMvc;

	@BeforeEach
	public void setUp() {
	}

	@Test
	@WithUserDetails("admin@tudelft.nl")
	public void testCoursesPage() throws Exception {
		this.mockMvc
				.perform(get("/courses"))
				.andExpect(status().isOk())
				.andExpect(model().attributeExists("courses"));
	}

	@Test
	@WithUserDetails("teacher1@tudelft.nl")
	public void testViewParticipants() throws Exception {
		Course course = courseRepository.findAll().get(0);
		this.mockMvc
				.perform(get("/course/" + course.getId() + "/participants"))
				.andExpect(status().isOk())
				.andExpect(model().attributeExists("staff"))
				.andExpect(model().attributeExists("students"));
	}

	@Test
	@WithUserDetails("student1@tudelft.nl")
	public void testCorrectRedirectIfNotinCourse() throws Exception {
		this.mockMvc.perform(get("/course/2"))
				.andExpect(status().is3xxRedirection())
				.andExpect(redirectedUrl("/course/2/enroll"));
	}

	@Test
	@WithUserDetails("student1@tudelft.nl")
	public void testCorrectRedirectIfNotAllowed() throws Exception {
		this.mockMvc.perform(get("/course/create"))
				.andExpect(status().is2xxSuccessful()).andExpect(view().name("error/thouShaltNotPass"));
	}

}
