/*
 * Queue - A Queueing system that can be used to handle labs in higher education
 * Copyright (C) 2016-2020  Delft University of Technology
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <https://www.gnu.org/licenses/>.
 */
package nl.tudelft.ewi.queue.controller;

import static org.assertj.core.api.Assertions.assertThat;
import static org.springframework.security.test.web.servlet.request.SecurityMockMvcRequestPostProcessors.csrf;
import static org.springframework.test.web.servlet.request.MockMvcRequestBuilders.get;
import static org.springframework.test.web.servlet.request.MockMvcRequestBuilders.post;
import static org.springframework.test.web.servlet.result.MockMvcResultMatchers.*;

import nl.tudelft.ewi.queue.QueueApplication;
import nl.tudelft.ewi.queue.model.Lab;
import nl.tudelft.ewi.queue.model.User;
import nl.tudelft.ewi.queue.repository.LabRepository;
import nl.tudelft.ewi.queue.repository.UserRepository;

import org.junit.jupiter.api.Test;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.boot.test.autoconfigure.web.servlet.AutoConfigureMockMvc;
import org.springframework.boot.test.context.SpringBootTest;
import org.springframework.security.test.context.support.WithUserDetails;
import org.springframework.test.annotation.DirtiesContext;
import org.springframework.test.web.servlet.MockMvc;
import org.springframework.transaction.annotation.Transactional;

@SpringBootTest(classes = QueueApplication.class)
@AutoConfigureMockMvc()
@DirtiesContext(classMode = DirtiesContext.ClassMode.BEFORE_CLASS)
public class LabControllerTest {
	@Autowired
	private MockMvc mockMvc;

	@Autowired
	private LabRepository labRepository;

	@Autowired
	private UserRepository userRepository;

	@Test
	@WithUserDetails("student1@tudelft.nl")
	public void testCorrectRedirectIfNotinCourse() throws Exception {
		this.mockMvc.perform(get("/lab/3"))
				.andExpect(status().is3xxRedirection())
				.andExpect(redirectedUrl("/course/2/enroll"));
	}

	@Test
	@WithUserDetails("student1@tudelft.nl")
	public void testCorrectRedirectIfNotAllowed() throws Exception {
		this.mockMvc.perform(get("/lab/1/edit"))
				.andExpect(status().is2xxSuccessful()).andExpect(view().name("error/thouShaltNotPass"));
	}

	@Test
	@WithUserDetails("manager2@tudelft.nl")
	@Transactional
	public void testCorrectUnenqueue() throws Exception {
		// Get a student which is already enrolled in lab 3.
		User student = userRepository.findById(1L).orElseThrow();
		// Get a lab with timeslots enabled
		Lab lab = labRepository.findById(3L).orElseThrow();
		this.mockMvc.perform(post("/lab/" + lab.getId() + "/unenqueue/" + student.getId())
				.with(csrf()))
				.andExpect(status().is3xxRedirection());
		assertThat(lab.isEnqueued(student)).isFalse();
	}

	@Test
	@WithUserDetails("teacher1@tudelft.nl")
	@Transactional
	public void testCloseEnqueueing() throws Exception {
		mockMvc.perform(post("/lab/2/close-enqueue/true").with(csrf()))
				.andExpect(status().is3xxRedirection())
				.andExpect(redirectedUrl("/lab/2"));
		Lab lab = labRepository.findByIdOrThrow(2L);
		assertThat(lab.isEnqueueClosed()).isTrue();
	}

	@Test
	@WithUserDetails("teacher1@tudelft.nl")
	@Transactional
	public void testOpenEnqueueing() throws Exception {
		Lab lab = labRepository.findByIdOrThrow(2L);
		lab.setEnqueueClosed(true);
		mockMvc.perform(post("/lab/2/close-enqueue/false").with(csrf()))
				.andExpect(status().is3xxRedirection())
				.andExpect(redirectedUrl("/lab/2"));
		assertThat(lab.isEnqueueClosed()).isFalse();
	}

}
