/*
 * Queue - A Queueing system that can be used to handle labs in higher education
 * Copyright (C) 2016-2020  Delft University of Technology
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <https://www.gnu.org/licenses/>.
 */
package nl.tudelft.ewi.queue.controller;

import static org.hamcrest.Matchers.contains;
import static org.hamcrest.Matchers.hasSize;
import static org.springframework.test.web.servlet.request.MockMvcRequestBuilders.get;
import static org.springframework.test.web.servlet.result.MockMvcResultMatchers.model;
import static org.springframework.test.web.servlet.result.MockMvcResultMatchers.status;

import nl.tudelft.ewi.queue.QueueApplication;
import nl.tudelft.ewi.queue.model.Course;
import nl.tudelft.ewi.queue.model.Notification;
import nl.tudelft.ewi.queue.model.User;
import nl.tudelft.ewi.queue.repository.CourseRepository;
import nl.tudelft.ewi.queue.repository.NotificationRepository;
import nl.tudelft.ewi.queue.repository.UserRepository;

import org.junit.jupiter.api.Test;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.boot.test.autoconfigure.web.servlet.AutoConfigureMockMvc;
import org.springframework.boot.test.context.SpringBootTest;
import org.springframework.security.test.context.support.WithUserDetails;
import org.springframework.test.annotation.DirtiesContext;
import org.springframework.test.web.servlet.MockMvc;

@SpringBootTest(classes = QueueApplication.class)
@AutoConfigureMockMvc
@DirtiesContext(classMode = DirtiesContext.ClassMode.BEFORE_CLASS)
public class NotificationControllerTest {

	@Autowired
	private NotificationRepository notificationRepository;

	@Autowired
	private CourseRepository courseRepository;

	@Autowired
	private UserRepository userRepository;

	@Autowired
	private MockMvc mockMvc;

	/**
	 * Test to make sure the default notifications page is working.
	 *
	 * @throws Exception
	 */
	@Test
	@WithUserDetails("student1@tudelft.nl")
	public void testNotificationsPage() throws Exception {
		this.mockMvc.perform(get("/notifications"))
				.andExpect(status().isOk());
	}

	/**
	 * Making sure that adding a notification to a student doesn't break the notifications page, and that the
	 * correct amount of notifications gets shown.
	 *
	 * @throws Exception
	 */
	@Test
	@WithUserDetails("student1@tudelft.nl")
	public void testFeedbackNotification() throws Exception {
		User student = userRepository.findByUsername("student1@tudelft.nl");
		Course course = courseRepository.findAll().get(0);
		Notification notification = new Notification(student, course, "test",
				"testing notification", 100, "url");
		notificationRepository.save(notification);
		this.mockMvc.perform(get("/notifications"))
				.andExpect(status().isOk())
				.andExpect(model().attributeExists("notifications"))
				.andExpect(model().attribute("notifications", hasSize(1)))
				.andExpect(model().attribute("notifications", contains(notification)));
	}
}
