/*
 * Queue - A Queueing system that can be used to handle labs in higher education
 * Copyright (C) 2016-2020  Delft University of Technology
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <https://www.gnu.org/licenses/>.
 */
package nl.tudelft.ewi.queue.controller;

import static org.assertj.core.api.Assertions.assertThat;
import static org.springframework.security.test.web.servlet.request.SecurityMockMvcRequestPostProcessors.csrf;
import static org.springframework.test.web.servlet.request.MockMvcRequestBuilders.get;
import static org.springframework.test.web.servlet.request.MockMvcRequestBuilders.post;
import static org.springframework.test.web.servlet.result.MockMvcResultMatchers.*;

import java.time.LocalDateTime;

import javax.transaction.Transactional;

import nl.tudelft.ewi.queue.QueueApplication;
import nl.tudelft.ewi.queue.model.*;
import nl.tudelft.ewi.queue.repository.AssignmentRepository;
import nl.tudelft.ewi.queue.repository.CourseRepository;
import nl.tudelft.ewi.queue.repository.RequestRepository;
import nl.tudelft.ewi.queue.repository.UserRepository;

import org.junit.jupiter.api.BeforeEach;
import org.junit.jupiter.api.Test;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.boot.test.autoconfigure.web.servlet.AutoConfigureMockMvc;
import org.springframework.boot.test.context.SpringBootTest;
import org.springframework.security.test.context.support.WithUserDetails;
import org.springframework.test.annotation.DirtiesContext;
import org.springframework.test.web.servlet.MockMvc;
import org.springframework.test.web.servlet.MvcResult;

import com.fasterxml.jackson.databind.ObjectMapper;

@SpringBootTest(classes = QueueApplication.class)
@AutoConfigureMockMvc
@DirtiesContext(classMode = DirtiesContext.ClassMode.BEFORE_CLASS)
public class RequestControllerTest {

	@Autowired
	private RequestController controller;

	@Autowired
	AssignmentRepository assignmentRepository;

	@Autowired
	RequestRepository requestRepository;

	@Autowired
	CourseRepository courseRepository;

	@Autowired
	UserRepository userRepository;

	@Autowired
	ObjectMapper objectMapper;

	@Autowired
	private MockMvc mockMvc;

	private Request request;

	@BeforeEach
	public void setUp() {
		Request templateRequest = requestRepository.findAll().iterator().next();

		request = new Request(templateRequest.getRequestEntity(), templateRequest.getAssignment(),
				templateRequest.getRoom(), templateRequest.getRequestType(), "", templateRequest.getLab());
		requestRepository.save(request);
	}

	@Test
	public void contextLoads() {
		assert controller != null;
	}

	@Test
	@WithUserDetails("admin@tudelft.nl")
	public void testRequestList() throws Exception {
		this.mockMvc
				.perform(get("/requests"))
				.andExpect(status().isOk())
				.andExpect(model().attributeExists("requests"));
	}

	@Test
	@WithUserDetails("teacher1@tudelft.nl")
	public void testViewRequest() throws Exception {
		this.mockMvc
				.perform(get("/request/" + request.getId()))
				.andExpect(status().isOk())
				.andExpect(model().attributeExists("request"));
	}

	@Test
	@WithUserDetails("teacher1@tudelft.nl")
	public void testApproveARequestGet() throws Exception {
		this.mockMvc
				.perform(get("/request/" + request.getId() + "/approve"))
				.andExpect(status().isOk())
				.andExpect(model().attributeExists("request"));
	}

	@Test
	@Transactional
	@WithUserDetails("teacher1@tudelft.nl")
	public void testApproveARequestPost() throws Exception {
		this.mockMvc
				.perform(
						post("/request/" + request.getId() + "?approve=")
								.with(csrf())
								.param("comment", "Well done"))
				.andExpect(status().is3xxRedirection())
				.andExpect(flash().attribute("message", "Request approved."));
	}

	@Test
	@Transactional
	@WithUserDetails("student1@tudelft.nl")
	public void testStudentCannotApproveOwnRequest() throws Exception {
		MvcResult result = mockMvc
				.perform(
						post("/request/" + request.getId() + "?approve=")
								.with(csrf())
								.param("comment", "Well done"))
				.andExpect(status().is2xxSuccessful()).andReturn();
		String pageContents = result.getResponse().getContentAsString();
		assertThat(pageContents.contains("Access Denied")).isTrue();
		assertThat(request.isApproved()).isFalse();
	}

	@Test
	@WithUserDetails("teacher1@tudelft.nl")
	public void testRejectARequestGet() throws Exception {
		this.mockMvc
				.perform(get("/request/" + request.getId() + "/reject"))
				.andExpect(status().isOk())
				.andExpect(model().attributeExists("request")).andReturn();
	}

	@Test
	@WithUserDetails("teacher1@tudelft.nl")
	public void testRejectARequestPost() throws Exception {
		User assistant = userRepository.findByUsername("assistant1@tudelft.nl");
		RequestProcessingEvent processing = new RequestProcessingEvent(request, LocalDateTime.now());
		RequestAssignedEvent assigned = new RequestAssignedEvent(request, LocalDateTime.now(), assistant);
		request.addEvent(processing);
		request.addEvent(assigned);
		requestRepository.save(request);

		mockMvc
				.perform(
						post("/request/" + request.getId() + "?reject=")
								.with(csrf())
								.param("comment", "FAIL FAIL FAIl")
								.param("commentForStudent", "FAIL FAIL FAIl"))
				.andExpect(status().is3xxRedirection()).andReturn();
		request = requestRepository.findById(request.getId()).orElseThrow();
		assertThat(request.isRejected()).isTrue();
	}

	@Test
	@WithUserDetails("teacher1@tudelft.nl")
	public void testForwardRequestGet() throws Exception {
		this.mockMvc
				.perform(get("/request/" + request.getId() + "/forward"))
				.andExpect(status().isOk())
				.andExpect(model().attributeExists("request")).andReturn();
	}

	@Test
	@Transactional
	@WithUserDetails("teacher1@tudelft.nl")
	public void testForwardRequestPost() throws Exception {
		User assistant = userRepository.findByUsername("assistant2@tudelft.nl");
		User assistant2 = userRepository.findByUsername("assistant3@tudelft.nl");
		request.setAssistant(assistant);
		requestRepository.save(request);

		mockMvc
				.perform(
						post("/request/" + request.getId() + "?forward=")
								.with(csrf())
								.param("assistant", assistant2.getId().toString())
								.param("comment", "What is this, I don't even"))
				.andExpect(status().is3xxRedirection())
				.andExpect(flash().attribute("message", "Request forwarded."));
		request = requestRepository.findById(request.getId()).orElseThrow();
		assertThat(request.isAssigned()).isTrue();
	}

	@Test
	@WithUserDetails("teacher1@tudelft.nl")
	public void testExportListRequests() throws Exception {
		Course course = courseRepository.findAll().iterator().next();
		MvcResult result = mockMvc
				.perform(get("/course/" + course.getId() + "/requests/export"))
				.andExpect(status().isOk())
				.andReturn();
		String resultJson = result.getResponse().getContentAsString();
		assertThat(resultJson.contains("student1")).isTrue();
		assertThat(resultJson.contains("Submission")).isTrue();
	}

	@Test
	@WithUserDetails("student1@tudelft.nl")
	public void testPostFeedbackOnRequest() throws Exception {
		mockMvc
				.perform(
						post("/request/" + request.getId() + "/feedback")
								.with(csrf())
								.param("feedback", "This TA did not even talk to me!"))
				.andExpect(status().is3xxRedirection())
				.andExpect(flash().attribute("message", "Feedback successfully saved."))
				.andReturn();
		request = requestRepository.findById(request.getId()).orElseThrow();
		assertThat(request.getFeedback()).isNotNull();
	}

	@Test
	@WithUserDetails("student2@tudelft.nl")
	public void testCannotPostFeedbackOnNotOwnRequest() throws Exception {
		request.setFeedback(null);
		requestRepository.save(request);
		MvcResult result = mockMvc
				.perform(
						post("/request/" + request.getId() + "/feedback")
								.with(csrf())
								.param("feedback", "This TA did not even talk to me!"))
				.andExpect(status().is2xxSuccessful())
				.andReturn();
		request = requestRepository.findById(request.getId()).orElseThrow();
		assertThat(result.getResponse().getContentAsString().contains("Oops. Something went wrong..."))
				.isTrue();
		assertThat(request.getFeedback()).isNull();
	}
}
