/*
 * Queue - A Queueing system that can be used to handle labs in higher education
 * Copyright (C) 2016-2020  Delft University of Technology
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <https://www.gnu.org/licenses/>.
 */
package nl.tudelft.ewi.queue.csv;

import static org.assertj.core.api.Assertions.assertThat;
import static org.junit.jupiter.api.Assertions.assertThrows;

import java.io.IOException;
import java.util.List;

import org.junit.jupiter.api.Test;
import org.springframework.mock.web.MockMultipartFile;

public class BrightspaceCsvHelperTest {

	@Test
	public void testFirstCsvEntity() throws IOException, EmptyCsvException, InvalidCsvException {

		String csv = "OrgDefinedId,Username,LastName,FirstName,Email,GroupCategory,GroupName\n" +
				"42,ww,willy,wonka,ww@tudelft.nl,categ,45";

		MockMultipartFile mock = new MockMultipartFile("test", csv.getBytes());
		List<BrightspaceCsvHelper> results = BrightspaceCsvHelper.readCsv(mock);
		BrightspaceCsvHelper first = results.get(0);

		BrightspaceCsvHelper expected = new BrightspaceCsvHelper(42, "ww", "willy", "wonka", "ww@tudelft.nl",
				"categ", "45");

		assertThat(expected).isEqualTo(first);
	}

	@Test
	public void testNameWithComma() throws IOException, EmptyCsvException, InvalidCsvException {

		String csv = "OrgDefinedId,Username,LastName,FirstName,Email,GroupCategory,GroupName\n" +
				"42,ww,willy,\"wonka, huan\",ww@tudelft.nl,category,45";

		MockMultipartFile mock = new MockMultipartFile("test", csv.getBytes());
		List<BrightspaceCsvHelper> results = BrightspaceCsvHelper.readCsv(mock);
		BrightspaceCsvHelper first = results.get(0);

		BrightspaceCsvHelper expected = new BrightspaceCsvHelper(42, "ww", "willy", "wonka, huan",
				"ww@tudelft.nl", "category", "45");

		assertThat(expected).isEqualTo(first);
	}

	@Test
	public void testEmptyCsv() {
		String csv = "";
		MockMultipartFile mock = new MockMultipartFile("test", csv.getBytes());

		assertThrows(EmptyCsvException.class, () -> {
			BrightspaceCsvHelper.readCsv(mock);
		});
	}

	@Test
	public void testTooManyDelimiters() {
		String csv = "OrgDefinedId,Username,LastName,FirstName,Email,GroupCategory,GroupName\n" +
				"asasa;asa;sa;s;as;as;as;as;as";
		MockMultipartFile mock = new MockMultipartFile("test", csv.getBytes());

		assertThrows(InvalidCsvException.class, () -> {
			BrightspaceCsvHelper.readCsv(mock);
		});
	}

	@Test
	public void testTooFewDelimiters() {
		String csv = "OrgDefinedId,Username,LastName,FirstName,Email,GroupCategory,GroupName\n" +
				"asasa;as";
		MockMultipartFile mock = new MockMultipartFile("test", csv.getBytes());

		assertThrows(InvalidCsvException.class, () -> {
			BrightspaceCsvHelper.readCsv(mock);
		});
	}
}
