/*
 * Queue - A Queueing system that can be used to handle labs in higher education
 * Copyright (C) 2016-2020  Delft University of Technology
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <https://www.gnu.org/licenses/>.
 */
package nl.tudelft.ewi.queue.csv;

import static org.assertj.core.api.Assertions.assertThat;
import static org.junit.jupiter.api.Assertions.assertThrows;

import java.io.IOException;
import java.util.List;

import nl.tudelft.ewi.queue.model.Assistant;
import nl.tudelft.ewi.queue.model.DefaultRole;
import nl.tudelft.ewi.queue.model.Manager;
import nl.tudelft.ewi.queue.model.Role;
import nl.tudelft.ewi.queue.model.Teacher;

import org.junit.jupiter.api.Test;
import org.springframework.mock.web.MockMultipartFile;

public class UserCsvHelperTest {

	@Test
	public void testFirstCsvEntity() throws IOException, EmptyCsvException, InvalidCsvException {
		String csv = "Name,role\n" + "Mara;ta\n Liam;ta";
		MockMultipartFile mock = new MockMultipartFile("test", csv.getBytes());
		List<UserCsvHelper> results = UserCsvHelper.readCsv(mock);
		UserCsvHelper first = results.get(0);
		assertThat(first.getNetId()).isEqualTo("Mara");
		assertThat(first.getRole()).isEqualByComparingTo(DefaultRole.ROLE_ASSISTANT);
	}

	@Test
	public void testTooManyDelimiters() {
		String csv = "Name,role\n" + "Mara;student;frauw\n";
		MockMultipartFile mock = new MockMultipartFile("test", csv.getBytes());

		assertThrows(InvalidCsvException.class, () -> UserCsvHelper.readCsv(mock));
	}

	@Test
	public void testLastCsvEntity() throws IOException, EmptyCsvException, InvalidCsvException {
		String csv = "Name,role\n" + "Mara;ta\n Liam;ta\nWillyWonka;admin";
		MockMultipartFile mock = new MockMultipartFile("test", csv.getBytes());
		List<UserCsvHelper> results = UserCsvHelper.readCsv(mock);
		UserCsvHelper third = results.get(2);

		assertThat(third.getNetId()).isEqualTo("WillyWonka");
		assertThat(third.getRole()).isEqualByComparingTo(DefaultRole.ROLE_ADMIN);
	}

	@Test
	public void testEmptyCsv() {
		String csv = "";
		MockMultipartFile mock = new MockMultipartFile("test", csv.getBytes());

		assertThrows(EmptyCsvException.class, () -> UserCsvHelper.readCsv(mock));
	}

	@Test
	public void testUnknownRole() {
		String csv = "Name,role\n" + "Mara;taw\n";
		MockMultipartFile mock = new MockMultipartFile("test", csv.getBytes());

		assertThrows(InvalidCsvException.class, () -> UserCsvHelper.readCsv(mock));
	}

	@Test
	public void testGetRoleAssistant() throws DefaultRole.UnknownRoleException {
		UserCsvHelper uch = new UserCsvHelper("netid", "ta");
		Role role = uch.getRole(null, null);

		assertThat(role.getClass()).isEqualTo(Assistant.class);
	}

	@Test
	public void testGetRoleManager() throws DefaultRole.UnknownRoleException {
		UserCsvHelper uch = new UserCsvHelper("netid", "manager");
		Role role = uch.getRole(null, null);

		assertThat(role.getClass()).isEqualTo(Manager.class);
	}

	@Test
	public void testGetRoleTeacher() throws DefaultRole.UnknownRoleException {
		UserCsvHelper uch = new UserCsvHelper("netid", "teacher");
		Role role = uch.getRole(null, null);

		assertThat(role.getClass()).isEqualTo(Teacher.class);
	}

	@Test
	public void testGetRoleUnexpected() throws DefaultRole.UnknownRoleException {
		UserCsvHelper uch = new UserCsvHelper("netid", "student");

		assertThrows(IllegalStateException.class, () -> uch.getRole(null, null));
	}
}
