/*
 * Queue - A Queueing system that can be used to handle labs in higher education
 * Copyright (C) 2016-2020  Delft University of Technology
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <https://www.gnu.org/licenses/>.
 */
package nl.tudelft.ewi.queue.model;

import static org.assertj.core.api.Assertions.assertThat;
import static org.mockito.Mockito.when;

import java.time.LocalDateTime;
import java.time.format.DateTimeFormatter;
import java.util.List;
import java.util.Optional;
import java.util.OptionalDouble;

import org.junit.jupiter.api.BeforeEach;
import org.junit.jupiter.api.Test;
import org.mockito.Mockito;

import com.fasterxml.jackson.core.JsonProcessingException;

public class CourseTest {

	private Course course;
	private Lab lab;

	@BeforeEach
	public void init() {
		course = new Course();
		lab = new Lab();
	}

	@Test
	public void getTodaysLabsEmptyCourse() {
		assertThat(course.getTodaysLabs().isEmpty()).isTrue();
	}

	@Test
	public void getTodaysLabsLabNotToday() {
		lab.setSlot(new LabSlot(LocalDateTime.now().minusYears(1), LocalDateTime.now().minusYears(1)));
		course.addLab(lab);
		assertThat(course.getTodaysLabs().isEmpty()).isTrue();
	}

	@Test
	public void getTodaysLabsLabNotOpen() {
		lab.setSlot(new LabSlot(LocalDateTime.now().minusYears(1), LocalDateTime.now().minusYears(1)));
		lab.setSignOffIntervals(false);
		course.addLab(lab);
		assertThat(course.getTodaysLabs().isEmpty()).isTrue();
	}

	@Test
	public void getTodaysLabsLabIsOpen() {
		lab.setSlot(new LabSlot(LocalDateTime.now().plusYears(1), LocalDateTime.now().plusYears(1)));
		lab.setSignOffIntervals(true);
		lab.setSlotSelectionOpensAt(LocalDateTime.now().minusYears(1));
		course.addLab(lab);
		assertThat(course.getTodaysLabs().isEmpty()).isFalse();
		assertThat(course.getTodaysLabs().contains(lab)).isTrue();
	}

	@Test
	public void getTodaysLabsLabIsToday() {
		lab.setSlot(new LabSlot(LocalDateTime.now(), LocalDateTime.now()));
		course.addLab(lab);
		assertThat(course.getTodaysLabs().isEmpty()).isFalse();
		assertThat(course.getTodaysLabs().contains(lab)).isTrue();
	}

	@Test
	public void getOldLabsEmptyCourse() {
		assertThat(course.getOldLabs().isEmpty()).isTrue();
	}

	@Test
	public void getOldLabsLabNotFinished() {
		lab.setSlot(new LabSlot(LocalDateTime.now().plusDays(1), LocalDateTime.now().plusDays(1)));
		course.addLab(lab);
		assertThat(course.getOldLabs().isEmpty()).isTrue();
	}

	@Test
	public void getOldLabsLabHasFinished() {
		lab.setSlot(new LabSlot(LocalDateTime.now().minusDays(1), LocalDateTime.now().minusDays(1)));
		course.addLab(lab);
		assertThat(course.getOldLabs().isEmpty()).isFalse();
		assertThat(course.getOldLabs().contains(lab)).isTrue();
	}

	@Test
	public void getActiveLabsEmptyCourse() {
		assertThat(course.getActiveLabs().isEmpty()).isTrue();
	}

	@Test
	public void getActiveLabsLabHasFinished() {
		lab.setSlot(new LabSlot(LocalDateTime.now().minusDays(1), LocalDateTime.now().minusDays(1)));
		course.addLab(lab);
		assertThat(course.getActiveLabs().isEmpty()).isTrue();
	}

	@Test
	public void getActiveLabsLabNotStarted() {
		lab.setSlot(new LabSlot(LocalDateTime.now().plusDays(1), LocalDateTime.now().plusDays(1)));
		course.addLab(lab);
		assertThat(course.getActiveLabs().isEmpty()).isTrue();
	}

	@Test
	public void getActiveLabsLabNotFinished() {
		lab.setSlot(new LabSlot(LocalDateTime.now().minusHours(1), LocalDateTime.now().plusHours(1)));
		course.addLab(lab);
		assertThat(course.getActiveLabs().isEmpty()).isFalse();
		assertThat(course.getActiveLabs().contains(lab)).isTrue();
	}

	@Test
	public void addLabNewLabSimple() {
		course.addLab(lab);
		assertThat(course.getLabs().contains(lab)).isTrue();
		assertThat(lab.getCourse()).isEqualTo(course);
	}

	@Test
	public void addLabSameCourseTwice() {
		course.addLab(lab);
		assertThat(lab.getCourse()).isEqualTo(course);
		assertThat(course.getLabs().contains(lab)).isTrue();

		course.addLab(lab);
		assertThat(course.getLabs().size()).isEqualTo(2);
	}

	@Test
	public void addAssignmentSimple() {
		Assignment assignment = new Assignment(course, "assignment");
		course.addAssignment(assignment);

		assertThat(course.getAssignments().isEmpty()).isFalse();
		assertThat(course.getAssignments().contains(assignment)).isTrue();
	}

	@Test
	public void addAssignmentSameCourseTwice() {
		Assignment assignment = new Assignment(new Course(), "assignment");
		course.addAssignment(assignment);

		assertThat(course.getAssignments().isEmpty()).isFalse();
		assertThat(course.getAssignments().contains(assignment)).isTrue();
		assertThat(assignment.getCourse()).isEqualTo(course);

		course.addAssignment(assignment);

		assertThat(course.getAssignments().size()).isEqualTo(2);
		assertThat(assignment.getCourse()).isEqualTo(course);
	}

	@Test
	public void addRoleSimple() {
		Role role = new Teacher(new User(), new Course());
		course.addRole(role);

		assertThat(course.getRoles().isEmpty()).isFalse();
		assertThat(course.getRoles().contains(role)).isTrue();
	}

	@Test
	public void addRoleSameCourseTwice() {
		Role role = new Teacher(new User(), new Course());
		course.addRole(role);

		assertThat(course.getRoles().isEmpty()).isFalse();
		assertThat(course.getRoles().contains(role)).isTrue();
		assertThat(role.getCourse()).isEqualTo(course);

		course.addRole(role);

		assertThat(course.getRoles().size()).isEqualTo(2);
		assertThat(role.getCourse()).isEqualTo(course);
	}

	@Test
	public void getPrivilegedEmptyCourse() {
		assertThat(course.getPrivileged().isEmpty()).isTrue();
	}

	@Test
	public void getPrivilegedStudent() {
		Role role = new Student(new User(), course, LocalDateTime.now());
		course.addRole(role);

		assertThat(course.getPrivileged().isEmpty()).isTrue();
	}

	@Test
	public void getPrivilegedAssistant() {
		Role role = new Assistant(new User(), course);
		course.addRole(role);

		assertThat(course.getPrivileged().isEmpty()).isFalse();
		assertThat(course.getPrivileged().contains(role.getUser())).isTrue();
	}

	@Test
	public void getPrivilegedManager() {
		Role role = new Manager(new User(), course);
		course.addRole(role);

		assertThat(course.getPrivileged().isEmpty()).isFalse();
		assertThat(course.getPrivileged().contains(role.getUser())).isTrue();
	}

	@Test
	public void getPrivilegedTeacher() {
		Role role = new Teacher(new User(), course);
		course.addRole(role);

		assertThat(course.getPrivileged().isEmpty()).isFalse();
		assertThat(course.getPrivileged().contains(role.getUser())).isTrue();
	}

	@Test
	public void addGroupSimple() {
		Group group = new Group();
		course.addGroup(group);

		assertThat(course.getGroups().isEmpty()).isFalse();
		assertThat(course.getGroups().contains(group)).isTrue();
	}

	@Test
	public void addGroupSameGroupTwice() {
		Group group = Mockito.mock(Group.class);
		when(group.getId()).thenReturn(1L);
		course.addGroup(group);

		assertThat(course.getGroups().size()).isEqualTo(1);
		assertThat(course.getGroups().contains(group)).isTrue();

		course.addGroup(group);
		assertThat(course.getGroups().size()).isEqualTo(1);
		assertThat(course.getGroups().contains(group)).isTrue();
	}

	@Test
	public void averageWaitingNoLabs() {
		assertThat(course.averageWaitingFormatted()).isEmpty();
	}

	@Test
	public void averageWaitingLabWithoutWaitingTime() {
		Lab mockLab = Mockito.mock(Lab.class);
		when(mockLab.getSlot()).thenReturn(new LabSlot(LocalDateTime.now(), LocalDateTime.now()));

		course.addLab(mockLab);

		assertThat(course.averageWaitingFormatted().isPresent()).isFalse();
	}

	@Test
	public void averageWaitingLabWithWaitingTime() {
		Lab mockLab = Mockito.mock(Lab.class);
		when(mockLab.getSlot()).thenReturn(new LabSlot(LocalDateTime.now(), LocalDateTime.now()));
		when(mockLab.averageWaitingDouble()).thenReturn(OptionalDouble.of(1.0));

		course.addLab(mockLab);

		assertThat(course.averageWaitingFormatted().isPresent()).isTrue();
		assertThat(course.averageWaitingFormatted()).isEqualTo(Optional.of("1"));
	}

	@Test
	public void averageProcessingNoLabs() {
		assertThat(course.averageProcessingFormatted()).isEmpty();
	}

	@Test
	public void averageProcessingLabWithoutWaitingTime() {
		Lab mockLab = Mockito.mock(Lab.class);
		when(mockLab.getSlot()).thenReturn(new LabSlot(LocalDateTime.now(), LocalDateTime.now()));

		course.addLab(mockLab);

		assertThat(course.averageProcessingFormatted().isPresent()).isFalse();
	}

	@Test
	public void averageProcessingLabWithWaitingTime() {
		Lab mockLab = Mockito.mock(Lab.class);
		when(mockLab.getSlot()).thenReturn(new LabSlot(LocalDateTime.now(), LocalDateTime.now()));
		when(mockLab.averageProcessingDouble()).thenReturn(OptionalDouble.of(1.0));

		course.addLab(mockLab);

		assertThat(course.averageProcessingFormatted().isPresent()).isTrue();
		assertThat(course.averageProcessingFormatted()).isEqualTo(Optional.of("1"));
	}

	@Test
	public void activeAssistantsNoLabs() {
		assertThat(course.activeAssistants()).isEqualTo(0);
	}

	@Test
	public void activeAssistantsLabWithActiveAssistant() {
		Lab mockLab = Mockito.mock(Lab.class);
		User assistant = new User();
		when(mockLab.getSlot()).thenReturn(new LabSlot(LocalDateTime.now(), LocalDateTime.now()));
		when(mockLab.activeAssistants()).thenReturn(List.of(assistant).stream());

		course.addLab(mockLab);

		assertThat(course.activeAssistants()).isEqualTo(1L);
	}

	@Test
	public void studentsInTheQueueNoLabs() {
		assertThat(course.studentsInTheQueue()).isEqualTo(0);
	}

	@Test
	public void studentsInTheQueueSimple() {
		Lab mockLab = Mockito.mock(Lab.class);
		when(mockLab.getSlot()).thenReturn(new LabSlot(LocalDateTime.now(), LocalDateTime.now()));
		when(mockLab.nrOfStudentsInQueue()).thenReturn(2L);

		course.addLab(mockLab);

		assertThat(course.studentsInTheQueue()).isEqualTo(2L);
	}

	@Test
	public void requestsTotalPerHourNoLabs() throws JsonProcessingException {
		assertThat(course.requestsTotalPerHour()).isEqualTo(
				"[[\"" + LocalDateTime.now().format(DateTimeFormatter.ofPattern("HH")) + "\"],[0]]");
	}

	@Test
	public void requestsTotalPerHourSimple() throws JsonProcessingException {
		Request request = Mockito.mock(Request.class);
		when(request.getCreatedAt()).thenReturn(LocalDateTime.now().minusMinutes(1));

		Lab mockLab = Mockito.mock(Lab.class);
		when(mockLab.getRequests()).thenReturn(List.of(request));
		when(mockLab.getSlot())
				.thenReturn(new LabSlot(LocalDateTime.now().minusHours(1), LocalDateTime.now()));

		course.addLab(mockLab);

		assertThat(course.requestsTotalPerHour()).isEqualTo(
				"[[\"" + LocalDateTime.now().minusHours(1).format(DateTimeFormatter.ofPattern("HH")) + "\",\""
						+ LocalDateTime.now().format(DateTimeFormatter.ofPattern("HH")) + "\"],[1,0]]");
	}

	@Test
	public void requestsRejectedPerHourNoLabs() throws JsonProcessingException {
		assertThat(course.requestsRejectedPerHour()).isEqualTo(
				"[[\"" + LocalDateTime.now().format(DateTimeFormatter.ofPattern("HH")) + "\"],[0]]");
	}

	@Test
	public void requestsRejectedPerHourSimple() throws JsonProcessingException {
		Request request = Mockito.mock(Request.class);
		when(request.getCreatedAt()).thenReturn(LocalDateTime.now().minusMinutes(1));

		Lab mockLab = Mockito.mock(Lab.class);
		when(mockLab.getRejected()).thenReturn(List.of(request));
		when(mockLab.getSlot())
				.thenReturn(new LabSlot(LocalDateTime.now().minusHours(1), LocalDateTime.now()));

		course.addLab(mockLab);

		assertThat(course.requestsRejectedPerHour()).isEqualTo(
				"[[\"" + LocalDateTime.now().minusHours(1).format(DateTimeFormatter.ofPattern("HH")) + "\",\""
						+ LocalDateTime.now().format(DateTimeFormatter.ofPattern("HH")) + "\"],[1,0]]");
	}

	@Test
	public void requestsApprovedPerHourNoLabs() throws JsonProcessingException {
		assertThat(course.requestsApprovedPerHour()).isEqualTo(
				"[[\"" + LocalDateTime.now().format(DateTimeFormatter.ofPattern("HH")) + "\"],[0]]");
	}

	@Test
	public void requestsApprovedPerHourSimple() throws JsonProcessingException {
		Request request = Mockito.mock(Request.class);
		when(request.getCreatedAt()).thenReturn(LocalDateTime.now().minusMinutes(1));

		Lab mockLab = Mockito.mock(Lab.class);
		when(mockLab.getApproved()).thenReturn(List.of(request));
		when(mockLab.getSlot())
				.thenReturn(new LabSlot(LocalDateTime.now().minusHours(1), LocalDateTime.now()));

		course.addLab(mockLab);

		assertThat(course.requestsApprovedPerHour()).isEqualTo(
				"[[\"" + LocalDateTime.now().minusHours(1).format(DateTimeFormatter.ofPattern("HH")) + "\",\""
						+ LocalDateTime.now().format(DateTimeFormatter.ofPattern("HH")) + "\"],[1,0]]");
	}

	@Test
	public void requestsHandledPerHourNoLabs() throws JsonProcessingException {
		assertThat(course.requestsHandledPerHour()).isEqualTo(
				"[[\"" + LocalDateTime.now().format(DateTimeFormatter.ofPattern("HH")) + "\"],[0]]");
	}

	@Test
	public void requestsHandledPerHourSimple() throws JsonProcessingException {
		Request request = Mockito.mock(Request.class);
		when(request.getCreatedAt()).thenReturn(LocalDateTime.now().minusMinutes(1));

		Lab mockLab = Mockito.mock(Lab.class);
		when(mockLab.getHandled()).thenReturn(List.of(request));
		when(mockLab.getSlot())
				.thenReturn(new LabSlot(LocalDateTime.now().minusHours(1), LocalDateTime.now()));

		course.addLab(mockLab);

		assertThat(course.requestsHandledPerHour()).isEqualTo(
				"[[\"" + LocalDateTime.now().minusHours(1).format(DateTimeFormatter.ofPattern("HH")) + "\",\""
						+ LocalDateTime.now().format(DateTimeFormatter.ofPattern("HH")) + "\"],[1,0]]");
	}

	@Test
	public void earliestLabOpensAtNotOpen() throws JsonProcessingException {
		lab.setSlot(new LabSlot(LocalDateTime.now().plusHours(1), LocalDateTime.now()));
		course.addLab(lab);

		course.requestsHandledPerHour();
		assertThat(course.requestsHandledPerHour()).isEqualTo(
				"[[\"" + LocalDateTime.now().format(DateTimeFormatter.ofPattern("HH")) + "\"],[0]]");
	}

	@Test
	public void earliestLabOpensAlreadyClosed() throws JsonProcessingException {
		lab.setSlot(new LabSlot(LocalDateTime.now(), LocalDateTime.now().plusMinutes(1)));
		course.addLab(lab);

		course.requestsHandledPerHour();
		assertThat(course.requestsHandledPerHour()).isEqualTo(
				"[[\"" + LocalDateTime.now().format(DateTimeFormatter.ofPattern("HH")) + "\"],[0]]");
	}

	@Test
	public void isDeletedFalse() {
		assertThat(course.isDeleted()).isFalse();
	}

}
