/*
 * Queue - A Queueing system that can be used to handle labs in higher education
 * Copyright (C) 2016-2020  Delft University of Technology
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <https://www.gnu.org/licenses/>.
 */
package nl.tudelft.ewi.queue.model;

import static org.assertj.core.api.Assertions.assertThat;

import java.time.LocalDateTime;
import java.util.Optional;

import nl.tudelft.ewi.queue.QueueApplication;

import org.junit.jupiter.api.BeforeEach;
import org.junit.jupiter.api.Test;
import org.springframework.boot.test.autoconfigure.web.servlet.AutoConfigureMockMvc;
import org.springframework.boot.test.context.SpringBootTest;
import org.springframework.test.annotation.DirtiesContext;
import org.springframework.test.context.web.WebAppConfiguration;

@SpringBootTest(classes = QueueApplication.class)
@WebAppConfiguration
@AutoConfigureMockMvc
@DirtiesContext(classMode = DirtiesContext.ClassMode.BEFORE_CLASS)
public class LabOnlineTest {

	Lab lab;

	@BeforeEach
	public void setup() {
		lab = new Lab();
	}

	@Test
	public void onlineTrue() {
		lab.setCommunicationMethod(CommunicationMethod.JITSI_MEET);
		assertThat(lab.isOnline()).isTrue();
	}

	@Test
	public void onlineFalseTAStudent() {
		lab.setCommunicationMethod(CommunicationMethod.TA_VISIT_STUDENT);
		assertThat(lab.isOnline()).isFalse();
	}

	@Test
	public void onlineFalseStudentTA() {
		lab.setCommunicationMethod(CommunicationMethod.STUDENT_VISIT_TA);
		assertThat(lab.isOnline()).isFalse();
	}

	@Test
	public void needsJitsiLink() {
		lab.setCommunicationMethod(CommunicationMethod.JITSI_MEET);

		Request request = new Request();
		request.setStatus(Request.Status.PROCESSING);
		request.setLab(lab);
		lab.addRequest(request);

		User user = new User();
		request.setRequestEntity(user);

		assertThat(lab.needsJitsiLink(user)).isTrue();
	}

	@Test
	public void getProcessingRequest() {
		Request request = new Request();
		request.setStatus(Request.Status.PROCESSING);
		request.setLab(lab);
		request.setCreatedAt(LocalDateTime.now());
		lab.addRequest(request);

		User user = new User();
		request.setRequestEntity(user);

		Optional<Request> requestOptional = lab.getProcessingRequest(user);
		assertThat(requestOptional).isPresent();
		assertThat(requestOptional.get()).isEqualByComparingTo(request);
	}
}
