/*
 * Queue - A Queueing system that can be used to handle labs in higher education
 * Copyright (C) 2016-2020  Delft University of Technology
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <https://www.gnu.org/licenses/>.
 */
package nl.tudelft.ewi.queue.model;

import static org.assertj.core.api.Assertions.assertThat;
import static org.junit.jupiter.api.Assertions.assertThrows;

import java.time.LocalDateTime;
import java.time.temporal.ChronoUnit;
import java.util.Collections;
import java.util.List;
import java.util.Optional;

import javax.transaction.Transactional;

import nl.tudelft.ewi.queue.QueueApplication;
import nl.tudelft.ewi.queue.exception.AlreadyEnqueuedException;
import nl.tudelft.ewi.queue.repository.LabRepository;
import nl.tudelft.ewi.queue.repository.RequestRepository;
import nl.tudelft.ewi.queue.repository.RequestTypeRepository;
import nl.tudelft.ewi.queue.repository.UserRepository;
import nl.tudelft.ewi.queue.service.LabService;

import org.junit.jupiter.api.BeforeEach;
import org.junit.jupiter.api.Test;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.boot.test.autoconfigure.web.servlet.AutoConfigureMockMvc;
import org.springframework.boot.test.context.SpringBootTest;
import org.springframework.test.annotation.DirtiesContext;
import org.springframework.test.context.web.WebAppConfiguration;

@SpringBootTest(classes = QueueApplication.class)
@WebAppConfiguration
@AutoConfigureMockMvc
@DirtiesContext(classMode = DirtiesContext.ClassMode.BEFORE_CLASS)
public class LabTest {

	@Autowired
	private RequestTypeRepository requestTypeRepository;

	@Autowired
	private RequestRepository requestRepository;

	@Autowired
	private LabRepository labRepository;

	@Autowired
	private UserRepository userRepository;

	@Autowired
	private LabService labService;

	private RequestType questionRequestType;

	private RequestType submissionRequestType;

	private Lab lab;

	private User user;

	@BeforeEach
	@Transactional
	public void setUp() {
		questionRequestType = new RequestType("Question");
		submissionRequestType = new RequestType("Submission");
		requestTypeRepository.save(questionRequestType);
		requestTypeRepository.save(submissionRequestType);
		lab = labRepository.findFirstByOrderByIdAsc();
		user = userRepository.findByUsername("student1@tudelft.nl");
	}

	@Test
	@Transactional
	public void testIsEnqueuedSingletonQueue() {
		lab.addRequest(new Request(user, null, new Room("DW 101"), questionRequestType, "", lab));
		assertThat(lab.isEnqueued(user)).isTrue();
	}

	@Test
	@Transactional
	public void testIsEnqueuedEmptyQueue() {
		// student chosen by random dice roll (also: above 25)
		User randomUser = userRepository.findByUsername("student42@tudelft.nl");
		assertThat(lab.isEnqueued(randomUser)).isFalse();
	}

	@Test
	@Transactional
	public void testPositionSingletonQueue() {
		lab.getQueue().add(new Request(user, null, new Room("DW 101"), submissionRequestType, "", lab));
		assertThat(lab.position(user)).isEqualTo("1");
	}

	@Test
	@Transactional
	public void testMultipleEnqueue() {
		Room room = new Room();

		Course course = new Course();

		Role role = new Student(user, course, LocalDateTime.now());
		course.addRole(role);
		user.addRole(role);

		Lab lab = new Lab(course, new LabSlot(LocalDateTime.now().minus(1,
				ChronoUnit.DAYS),
				LocalDateTime.now().plus(1,
						ChronoUnit.DAYS)),
				Collections.emptyList(), false);
		course.addLab(lab);

		Assignment assignment = new Assignment(course, "");
		lab.addAssignment(assignment);

		// Act
		Request request = new Request(user, assignment, room, submissionRequestType, "", lab);
		lab.addRequest(request);

		assertThrows(AlreadyEnqueuedException.class, () -> labService.enqueue(request));
	}

	@Test
	@Transactional
	public void testEnqueueWithoutEnroll() {
		User randomUser = userRepository.findByUsername("student43@tudelft.nl"); //again... random + 1
		assertThat(lab.isEnqueued(randomUser)).isFalse();
		assertThat(lab.getCourse().isEnrolled(randomUser)).isFalse();

		Request request = new Request(user, null, new Room("DW 101"), submissionRequestType, "", lab);
		lab.addRequest(request);
		assertThat(lab.position(user)).isEqualTo("1");
	}

	@Test
	@Transactional
	public void testGetQueue() {
		List<Request> queue = lab.getQueue();
		assertThat(queue).isNotNull();
		assertThat(queue.size()).isEqualTo(25); // adapted to state in DatabaseLoader
	}

	@Test
	@Transactional
	public void testGetPending() {
		List<Request> pending = lab.getPending();
		assertThat(pending).isNotNull();
		assertThat(pending.size()).isEqualTo(25); // adapted to state in DatabaseLoader
		assertThat(pending.get(0).getRequestEntity().getDisplayName()).isEqualTo("student1");
	}

	@Test
	@Transactional
	public void testGetProcessing() {
		List<Request> processing = lab.getProcessing();
		assertThat(processing).isNotNull();
		assertThat(processing.size()).isEqualTo(0);

		Request request = lab.getQueue().get(0);
		RequestEvent processingEvent = new RequestProcessingEvent(request, LocalDateTime.now());
		request.addEvent(processingEvent);
		requestRepository.save(request);
		processing = lab.getProcessing();
		assertThat(processing.size()).isEqualTo(1);
	}

	@Test
	@Transactional
	public void testGetArchived() {
		List<Request> archived = lab.getArchived();
		assertThat(archived).isNotNull();
		assertThat(archived.size()).isEqualTo(0);

		Request request = lab.getQueue().get(0);
		User assistant = userRepository.findByUsername("student5@tudelft.nl");
		RequestEvent handledEvent = new RequestApprovedEvent(request, assistant, LocalDateTime.now(), "Top!");
		request.addEvent(handledEvent);
		requestRepository.save(request);
		archived = lab.getArchived();
		assertThat(archived.size()).isEqualTo(1);
	}

	@Test
	@Transactional
	public void testGetHandled() {
		List<Request> handled = lab.getHandled();
		assertThat(handled).isNotNull();
		assertThat(handled.size()).isEqualTo(0);

		Request request = lab.getQueue().get(0);
		User assistant = userRepository.findByUsername("student5@tudelft.nl");
		RequestEvent approvedRequest = new RequestApprovedEvent(request, assistant, LocalDateTime.now(),
				"Top!");
		request.addEvent(approvedRequest);
		requestRepository.save(request);
		handled = lab.getHandled();
		assertThat(handled.size()).isEqualTo(1);
	}

	@Test
	@Transactional
	public void testGetApproved() {
		List<Request> approved = lab.getApproved();
		assertThat(approved).isNotNull();
		assertThat(approved.size()).isEqualTo(0);

		Request request = lab.getQueue().get(0);
		User assistant = userRepository.findByUsername("student5@tudelft.nl");
		RequestEvent approvedEvent = new RequestApprovedEvent(request, assistant, LocalDateTime.now(),
				"Top!");
		request.addEvent(approvedEvent);
		requestRepository.save(request);
		approved = lab.getApproved();
		assertThat(approved.size()).isEqualTo(1);
	}

	@Test
	@Transactional
	public void testGetRejected() {
		List<Request> rejected = lab.getRejected();
		assertThat(rejected).isNotNull();
		assertThat(rejected.size()).isEqualTo(0);

		User assistant = userRepository.findByUsername("student5@tudelft.nl");

		Request rejectedRequest = lab.getQueue().get(0);
		requestRepository.save(rejectedRequest);
		RequestEvent rejectedEvent = new RequestRejectedEvent(rejectedRequest, assistant, LocalDateTime.now(),
				"Fail!", "Fail!");
		rejectedRequest.addEvent(rejectedEvent);
		requestRepository.save(rejectedRequest);

		rejected = lab.getRejected();
		assertThat(rejected.size()).isEqualTo(1);
	}

	@Test
	@Transactional
	public void testGetPendingRequestForUser() {
		Optional<Request> pendingForUser = lab.getPendingRequest(user);
		assertThat(pendingForUser).isPresent();
		assertThat(pendingForUser.get().getRequestEntity()).isEqualTo(user);
	}

}
