/*
 * Queue - A Queueing system that can be used to handle labs in higher education
 * Copyright (C) 2016-2020  Delft University of Technology
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <https://www.gnu.org/licenses/>.
 */
package nl.tudelft.ewi.queue.model;

import static org.assertj.core.api.Assertions.assertThat;
import static org.junit.jupiter.api.Assertions.*;
import static org.mockito.Mockito.when;

import java.time.LocalDateTime;
import java.time.format.DateTimeFormatter;
import java.util.*;

import org.junit.jupiter.api.BeforeEach;
import org.junit.jupiter.api.Test;
import org.mockito.Mockito;

import com.fasterxml.jackson.core.JsonProcessingException;

public class RequestTest {

	private Request request;

	@BeforeEach
	public void init() {
		RequestEntity requestEntity = Mockito.mock(RequestEntity.class);
		Assignment assignment = Mockito.mock(Assignment.class);
		Room room = Mockito.mock(Room.class);
		RequestType requestType = Mockito.mock(RequestType.class);
		Lab lab = Mockito.mock(Lab.class);
		request = new Request(requestEntity, assignment, room, requestType, "", lab);
	}

	@Test
	public void setJitsiRoom() {
		LocalDateTime time = LocalDateTime.now();
		request.setCreatedAt(time);

		String displayName = "name surname";
		User user = new User();
		user.setDisplayName(displayName);
		request.setRequestEntity(user);

		String result = "namesurname" + time.format(DateTimeFormatter.ofPattern("ddmmHHMMss"));

		request.setJitsiRoom();

		assertThat(request.getJitsiRoom()).isEqualTo(result);
	}

	@Test
	public void displayJitsiRoom() {
		when(request.getLab().isOnline()).thenReturn(true);

		Long id1 = 4201L;
		User a1 = new User();
		a1.id = id1;

		Long id2 = 4202L;
		User a2 = new User();
		a2.id = id2;

		assertFalse(request.shouldDisplayJitsiRoom(a1));
		assertFalse(request.shouldDisplayJitsiRoom(a2));

		request.setAssistant(a1);

		assertFalse(request.shouldDisplayJitsiRoom(a2));
		assertTrue(request.shouldDisplayJitsiRoom(a1));
	}

	@Test
	public void displayJitsiRoomNotOnline() {
		when(request.getLab().isOnline()).thenReturn(false);

		Long id1 = 4201L;
		User a1 = new User();
		a1.id = id1;

		request.setAssistant(a1);

		assertFalse(request.shouldDisplayJitsiRoom(a1));
	}

	@Test
	public void calculateWaitingTime() {
		request.setStatus(Request.Status.APPROVED);
		request.setCreatedAt(LocalDateTime.now().minusMinutes(1));
		request.setProcessingAt(LocalDateTime.now());

		assertThat(request.waitingTime()).isEqualTo(1);
	}

	@Test
	public void requestWithNoWaitingTime() {
		request.setStatus(Request.Status.APPROVED);
		request.setCreatedAt(LocalDateTime.now().minusMinutes(1));
		request.setApprovedAt(LocalDateTime.now());

		assertThat(request.waitingTime()).isEqualTo(1);
	}

	@Test
	public void calculateProcessingTime() {
		request.setStatus(Request.Status.APPROVED);
		request.setCreatedAt(LocalDateTime.now().minusMinutes(2));
		request.setProcessingAt(LocalDateTime.now().minusSeconds(1));
		request.setApprovedAt(LocalDateTime.now());

		assertThat(request.waitingTime()).isEqualTo(1);
	}

	@Test
	public void requestWithNoProcessingTime() {
		request.setStatus(Request.Status.APPROVED);
		request.setCreatedAt(LocalDateTime.now().minusMinutes(1));
		request.setApprovedAt(LocalDateTime.now());

		assertThat(request.processingTime()).isEqualTo(1);
	}

	@Test
	public void setLabOtherLab() {
		Lab lab = new Lab();

		request.setLab(lab);
		assertThat(request.getLab()).isEqualTo(lab);
	}

	@Test
	public void setLabSameLabTwice() {
		Lab lab = new Lab();

		request.setLab(lab);
		assertThat(request.getLab()).isEqualTo(lab);
		assertThat(lab.getRequests().size()).isEqualTo(1);

		request.setLab(lab);
		assertThat(request.getLab()).isEqualTo(lab);
		assertThat(lab.getRequests().size()).isEqualTo(1);
	}

	@Test
	public void hasFeedbackNoFeedback() {
		request.setFeedback(null);
		assertThat(request.hasFeedback()).isFalse();
	}

	@Test
	public void hasFeedbackEmptyFeedback() {
		request.setFeedback("");
		assertThat(request.hasFeedback()).isFalse();
	}

	@Test
	public void hasFeedbackSimpleTest() {
		request.setFeedback("feedback");
		assertThat(request.hasFeedback()).isTrue();
	}

	@Test
	public void getSlotSentenceSameDay() {
		RequestSlot requestSlot = new RequestSlot(LocalDateTime.now(), LocalDateTime.now());
		request.setSlot(requestSlot);
		DateTimeFormatter formatter = DateTimeFormatter.ofPattern("HH:mm");
		LocalDateTime opens = requestSlot.getOpensAt();
		LocalDateTime closes = requestSlot.getClosesAt();
		assertThat(request.getSlotSentence()).isEqualTo(formatter.format(opens) + " - "
				+ formatter.format(closes));
	}

	@Test
	public void getSlotSentenceDifferentDay() {
		RequestSlot requestSlot = new RequestSlot(LocalDateTime.of(2020, 1, 1, 23, 30),
				LocalDateTime.of(2020, 1, 2, 0, 30));
		request.setSlot(requestSlot);
		assertThat(request.getSlotSentence()).isEqualTo("23:30 - 00:30 on 2 Jan 2020");
	}

	@Test
	public void getHandledAtApproved() {
		LocalDateTime now = LocalDateTime.now();
		request.setStatus(Request.Status.APPROVED);
		request.setApprovedAt(now);

		assertThat(request.getHandledAt()).isEqualToIgnoringSeconds(now);
	}

	@Test
	public void getHandledAtRejected() {
		LocalDateTime now = LocalDateTime.now();
		request.setStatus(Request.Status.REJECTED);
		request.setRejectedAt(now);

		assertThat(request.getHandledAt()).isEqualToIgnoringSeconds(now);
	}

	@Test
	public void getHandledAtForwarded() {
		LocalDateTime now = LocalDateTime.now();
		request.setStatus(Request.Status.FORWARDED);
		request.setForwardedAt(now);

		assertThat(request.getHandledAt()).isEqualToIgnoringSeconds(now);
	}

	@Test
	public void getHandledAtNotFound() {
		LocalDateTime now = LocalDateTime.now();
		request.setStatus(Request.Status.NOTFOUND);
		request.setNotFoundAt(now);

		assertThat(request.getHandledAt()).isEqualToIgnoringSeconds(now);
	}

	@Test
	public void getHandledAtNoStatus() {
		assertThat(request.getHandledAt()).isNull();
	}

	@Test
	public void getArchivedAtApproved() {
		LocalDateTime now = LocalDateTime.now();
		request.setStatus(Request.Status.APPROVED);
		request.setApprovedAt(now);

		assertThat(request.getArchivedAt()).isEqualToIgnoringSeconds(now);
	}

	@Test
	public void getArchivedAtRejected() {
		LocalDateTime now = LocalDateTime.now();
		request.setStatus(Request.Status.REJECTED);
		request.setRejectedAt(now);

		assertThat(request.getArchivedAt()).isEqualToIgnoringSeconds(now);
	}

	@Test
	public void getArchivedAtForwarded() {
		LocalDateTime now = LocalDateTime.now();
		request.setStatus(Request.Status.FORWARDED);
		request.setForwardedAt(now);

		assertThat(request.getArchivedAt()).isEqualToIgnoringSeconds(now);
	}

	@Test
	public void getArchivedAtNotFound() {
		LocalDateTime now = LocalDateTime.now();
		request.setStatus(Request.Status.NOTFOUND);
		request.setNotFoundAt(now);

		assertThat(request.getArchivedAt()).isEqualToIgnoringSeconds(now);
	}

	@Test
	public void getArchivedAtRevoked() {
		LocalDateTime now = LocalDateTime.now();
		request.setStatus(Request.Status.REVOKED);
		request.setRevokedAt(now);

		assertThat(request.getArchivedAt()).isEqualToIgnoringSeconds(now);
	}

	@Test
	public void getArchivedAtCancelled() {
		LocalDateTime now = LocalDateTime.now();
		request.setStatus(Request.Status.CANCELLED);
		request.setRevokedAt(now);

		assertThat(request.getArchivedAt()).isEqualToIgnoringSeconds(now);
	}

	@Test
	public void getArchivedAtNoStatus() {
		assertThat(request.getArchivedAt()).isNull();
	}

	@Test
	public void waitingTimeNotPending() {
		request.setStatus(Request.Status.PENDING);
		assertThrows(IllegalStateException.class, () -> request.waitingTime());
	}

	@Test
	public void waitingTimeProcessingTime() {
		LocalDateTime created = LocalDateTime.of(2020, 1, 1, 0, 0);
		LocalDateTime processing = LocalDateTime.of(2020, 1, 1, 1, 0);

		request.setStatus(Request.Status.APPROVED);
		request.setCreatedAt(created);
		request.setProcessingAt(processing);

		assertThat(request.waitingTime()).isEqualTo(60);
	}

	@Test
	public void waitingTimeApprovedTime() {
		LocalDateTime created = LocalDateTime.of(2020, 1, 1, 0, 0);
		LocalDateTime approved = LocalDateTime.of(2020, 1, 1, 1, 0);

		request.setStatus(Request.Status.APPROVED);
		request.setCreatedAt(created);
		request.setApprovedAt(approved);

		assertThat(request.waitingTime()).isEqualTo(60);
	}

	@Test
	public void waitingTimeNoProcessingTime() {
		LocalDateTime created = LocalDateTime.of(2020, 1, 1, 0, 0);

		request.setStatus(Request.Status.APPROVED);
		request.setCreatedAt(created);

		assertThat(request.waitingTime()).isEqualTo(0);
	}

	@Test
	public void processingTimeNotPending() {
		request.setStatus(Request.Status.PENDING);
		assertThrows(IllegalStateException.class, () -> request.processingTime());
	}

	@Test
	public void processingTimeNoProcessing() {
		LocalDateTime created = LocalDateTime.of(2020, 1, 1, 0, 0);
		LocalDateTime approved = LocalDateTime.of(2020, 1, 1, 1, 0);

		request.setStatus(Request.Status.APPROVED);
		request.setCreatedAt(created);
		request.setApprovedAt(approved);

		assertThat(request.processingTime()).isEqualTo(60);
	}

	@Test
	public void processingTimeProcessingTime() {
		LocalDateTime approved = LocalDateTime.of(2020, 1, 1, 1, 0);
		LocalDateTime processing = LocalDateTime.of(2020, 1, 1, 0, 0);

		request.setStatus(Request.Status.APPROVED);
		request.setApprovedAt(approved);
		request.setProcessingAt(processing);

		assertThat(request.processingTime()).isEqualTo(60);
	}

	@Test
	public void getPreviousDifferentAssignment() {
		Request r = new Request();
		r.setRequestEntity(request.getRequestEntity());

		r.setAssignment(new Assignment(new Course(), "new"));
		request.setAssignment(new Assignment(new Course(), "new assignment"));

		assertThat(request.getPrevious().isEmpty()).isTrue();
	}

	@Test
	public void isPendingStatusPending() {
		request.setStatus(Request.Status.PENDING);
		assertThat(request.isPending()).isTrue();
	}

	@Test
	public void isPendingStatusPicked() {
		request.setStatus(Request.Status.PICKED);
		assertThat(request.isPending()).isTrue();
	}

	@Test
	public void isPendingFalse() {
		request.setStatus(Request.Status.APPROVED);
		assertThat(request.isPending()).isFalse();
	}

	@Test
	public void isArchivedApproved() {
		request.setStatus(Request.Status.APPROVED);
		assertThat(request.isArchived()).isTrue();
	}

	@Test
	public void isArchivedRejected() {
		request.setStatus(Request.Status.REJECTED);
		assertThat(request.isArchived()).isTrue();
	}

	@Test
	public void isArchivedForwarded() {
		request.setStatus(Request.Status.FORWARDED);
		assertThat(request.isArchived()).isTrue();
	}

	@Test
	public void isArchivedRevoked() {
		request.setStatus(Request.Status.REVOKED);
		assertThat(request.isArchived()).isTrue();
	}

	@Test
	public void isArchivedCancelled() {
		request.setStatus(Request.Status.CANCELLED);
		assertThat(request.isArchived()).isTrue();
	}

	@Test
	public void isArchivedNotFound() {
		request.setStatus(Request.Status.NOTFOUND);
		assertThat(request.isArchived()).isTrue();
	}

	@Test
	public void isArchivedFalse() {
		request.setStatus(Request.Status.PENDING);
		assertThat(request.isArchived()).isFalse();
	}

	@Test
	public void isHandledApproved() {
		request.setStatus(Request.Status.APPROVED);
		assertThat(request.isHandled()).isTrue();
	}

	@Test
	public void isHandledRejected() {
		request.setStatus(Request.Status.REJECTED);
		assertThat(request.isHandled()).isTrue();
	}

	@Test
	public void isHandledForwarded() {
		request.setStatus(Request.Status.FORWARDED);
		assertThat(request.isHandled()).isTrue();
	}

	@Test
	public void isHandledNotFound() {
		request.setStatus(Request.Status.NOTFOUND);
		assertThat(request.isHandled()).isTrue();
	}

	@Test
	public void isHandledFalse() {
		request.setStatus(Request.Status.PENDING);
		assertThat(request.isHandled()).isFalse();
	}

	@Test
	public void isQueuedAssigned() {
		request.setStatus(Request.Status.ASSIGNED);
		assertThat(request.isQueued()).isTrue();
	}

	@Test
	public void isQueuedPending() {
		request.setStatus(Request.Status.PENDING);
		assertThat(request.isQueued()).isTrue();
	}

	@Test
	public void isQueuedFalse() {
		request.setStatus(Request.Status.APPROVED);
		assertThat(request.isQueued()).isFalse();
	}

	@Test
	public void equalsSameObject() {
		assertThat(request.equals(request)).isTrue();
	}

	@Test
	public void equalsNull() {
		assertThat(request.equals(null)).isFalse();
	}

	@Test
	public void equalsOtherClass() {
		assertThat(request.equals(new Course())).isFalse();
	}

	@Test
	public void equalsSameId() {
		Request r = new Request();
		assertThat(request.equals(r)).isTrue();
	}

	@Test
	public void compareToCreatedAfter() {
		request.setCreatedAt(LocalDateTime.now());
		Request r = new Request();
		r.setCreatedAt(LocalDateTime.now().plusMinutes(1));

		assertThat(request.compareTo(r)).isEqualTo(-1);
	}

	@Test
	public void compareToCreatedBefore() {
		request.setCreatedAt(LocalDateTime.now());
		Request r = new Request();
		r.setCreatedAt(LocalDateTime.now().minusMinutes(1));

		assertThat(request.compareTo(r)).isEqualTo(1);
	}

	@Test
	public void compareToCreatedSameTime() {
		LocalDateTime now = LocalDateTime.now();
		Request r = new Request();
		request.setCreatedAt(now);
		r.setCreatedAt(now);

		assertThat(request.compareTo(r)).isEqualTo(0);
	}

	@Test
	public void compareToOneNoSlot() {
		request.setSlot(new RequestSlot(LocalDateTime.now(), LocalDateTime.now()));
		Request r = new Request();
		request.setCreatedAt(LocalDateTime.now());
		r.setCreatedAt(LocalDateTime.now().plusMinutes(1));

		assertThat(request.compareTo(r)).isEqualTo(-1);
	}

	@Test
	public void compareToCompareSlots() {
		request.setSlot(new RequestSlot(LocalDateTime.now(), LocalDateTime.now()));
		Request r = new Request();
		r.setSlot(new RequestSlot(LocalDateTime.now().plusMinutes(1), LocalDateTime.now().plusMinutes(1)));

		assertThat(request.compareTo(r)).isEqualTo(-1);
	}

	@Test
	public void displayRequestSignOffIntervals() {
		Lab lab = new Lab();
		lab.setSignOffIntervals(false);
		request.setLab(lab);

		assertThat(request.displayRequest()).isTrue();
	}

	@Test
	public void displayRequestSlotNull() {
		Lab lab = new Lab();
		lab.setSignOffIntervals(true);
		lab.setCommunicationMethod(CommunicationMethod.JITSI_MEET);
		request.setLab(lab);

		assertThat(request.displayRequest()).isFalse();
	}

	@Test
	public void displayRequestOpensAtNull() {
		Lab lab = new Lab();
		lab.setSignOffIntervals(true);
		lab.setCommunicationMethod(CommunicationMethod.JITSI_MEET);
		request.setLab(lab);
		request.setSlot(new RequestSlot(null, null));

		assertThat(request.displayRequest()).isFalse();
	}

	@Test
	public void displayRequestBeforeInterval() {
		Lab lab = new Lab();
		lab.setSignOffIntervals(true);
		lab.setCommunicationMethod(CommunicationMethod.JITSI_MEET);
		request.setLab(lab);
		request.setSlot(new RequestSlot(LocalDateTime.now(), null));

		assertThat(request.displayRequest()).isTrue();
	}

	@Test
	public void requestsPerHourForSimple() throws JsonProcessingException {
		LocalDateTime opensAt = LocalDateTime.of(2020, 1, 1, 0, 0);
		LocalDateTime closesAt = LocalDateTime.of(2020, 1, 2, 0, 0);

		Request r1 = new Request();
		r1.setCreatedAt(opensAt);

		Request r2 = new Request();
		r2.setCreatedAt(closesAt.minusMinutes(1));

		Request r3 = new Request();
		r3.setCreatedAt(opensAt.plusMinutes(1));

		List<Request> requests = Arrays.asList(request, r1, r2, r3);
		String value = Request.requestsPerHour(opensAt, closesAt, requests);

		assertThat(value).isEqualTo(
				"[[\"00\",\"01\",\"02\",\"03\",\"04\",\"05\",\"06\",\"07\",\"08\",\"09\",\"10\",\"11\",\"12\",\"13\",\"14\",\"15\",\"16\",\"17\",\"18\",\"19\",\"20\",\"21\",\"22\",\"23\"],[2,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,1]]");
	}

	@Test
	public void hasFeedbackRatingNull() {
		assertThat(request.hasFeedbackRating()).isFalse();
	}

	@Test
	public void hasFeedbackRatingTrue() {
		request.setFeedbackRating(1);
		assertThat(request.hasFeedbackRating()).isTrue();
	}

	@Test
	public void isRevokedFalse() {
		request.setStatus(Request.Status.PENDING);
		assertThat(request.isRevoked()).isFalse();
	}

	@Test
	public void isRevokedTrue() {
		request.setStatus(Request.Status.REVOKED);
		assertThat(request.isRevoked()).isTrue();
	}

	@Test
	public void isProcessingFalse() {
		request.setStatus(Request.Status.PENDING);
		assertThat(request.isProcessing()).isFalse();
	}

	@Test
	public void isProcessingTrue() {
		request.setStatus(Request.Status.PROCESSING);
		assertThat(request.isProcessing()).isTrue();
	}

	@Test
	public void isApprovedFalse() {
		request.setStatus(Request.Status.PENDING);
		assertThat(request.isApproved()).isFalse();
	}

	@Test
	public void isApprovedTrue() {
		request.setStatus(Request.Status.APPROVED);
		assertThat(request.isApproved()).isTrue();
	}

	@Test
	public void isRejectedFalse() {
		request.setStatus(Request.Status.PENDING);
		assertThat(request.isRejected()).isFalse();
	}

	@Test
	public void isRejectedTrue() {
		request.setStatus(Request.Status.REJECTED);
		assertThat(request.isRejected()).isTrue();
	}

	@Test
	public void isForwardedFalse() {
		request.setStatus(Request.Status.PENDING);
		assertThat(request.isForwarded()).isFalse();
	}

	@Test
	public void isForwardedTrue() {
		request.setStatus(Request.Status.FORWARDED);
		assertThat(request.isForwarded()).isTrue();
	}

	@Test
	public void isNotFoundFalse() {
		request.setStatus(Request.Status.PENDING);
		assertThat(request.isNotFound()).isFalse();
	}

	@Test
	public void isNotFoundTrue() {
		request.setStatus(Request.Status.NOTFOUND);
		assertThat(request.isNotFound()).isTrue();
	}

	@Test
	public void isPickedFalse() {
		request.setStatus(Request.Status.PENDING);
		assertThat(request.isPicked()).isFalse();
	}

	@Test
	public void isPickedTrue() {
		request.setStatus(Request.Status.PICKED);
		assertThat(request.isPicked()).isTrue();
	}

}
