/*
 * Queue - A Queueing system that can be used to handle labs in higher education
 * Copyright (C) 2016-2020  Delft University of Technology
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <https://www.gnu.org/licenses/>.
 */
package nl.tudelft.ewi.queue.model;

import static org.assertj.core.api.Assertions.assertThat;

import java.time.LocalDateTime;

import org.junit.jupiter.api.BeforeEach;
import org.junit.jupiter.api.Test;

public class SlotTest {

	private Slot slot;
	private Slot other;

	@BeforeEach
	public void init() {
		slot = new LabSlot(LocalDateTime.now(), LocalDateTime.now().plusHours(1));
		slot.setId(1L);
		other = new LabSlot(slot.getOpensAt(), slot.getClosesAt());
		other.setId(1L);
	}

	@Test
	public void containsTimeBeforeOpens() {
		LocalDateTime time = LocalDateTime.now().minusHours(1);
		assertThat(slot.contains(time)).isFalse();
	}

	@Test
	public void containsTimeAfterCloses() {
		LocalDateTime time = LocalDateTime.now().plusHours(2);
		assertThat(slot.contains(time)).isFalse();
	}

	@Test
	public void containsTimeTrue() {
		LocalDateTime time = LocalDateTime.now().plusMinutes(30);
		assertThat(slot.contains(time)).isTrue();
	}

	@Test
	public void equalsSameObject() {
		assertThat(slot.equals(slot)).isTrue();
	}

	@Test
	public void equalsNull() {
		assertThat(slot.equals(null)).isFalse();
	}

	@Test
	public void equalsOtherInstance() {
		assertThat(slot.equals(new Course())).isFalse();
	}

	@Test
	public void equalsDifferentId() {
		other.setId(2L);
		assertThat(slot.equals(other)).isFalse();
	}

	@Test
	public void equalsSimple() {
		assertThat(slot.equals(other)).isTrue();
	}

	@Test
	public void compareToBefore() {
		other.setOpensAt(other.getOpensAt().minusHours(1));
		assertThat(slot.compareTo(other)).isEqualTo(1);
	}

	@Test
	public void compareToAfter() {
		other.setOpensAt(other.getOpensAt().plusHours(1));
		assertThat(slot.compareTo(other)).isEqualTo(-1);
	}

	@Test
	public void compareToSame() {
		assertThat(slot.compareTo(other)).isEqualTo(0);
	}

	@Test
	public void isTodayishClosesAtLater() {
		slot.setClosesAt(slot.getClosesAt().minusDays(2));
		assertThat(slot.isTodayish()).isFalse();
	}

	@Test
	public void isTodayishOpensAtEarlier() {
		slot.setOpensAt(slot.getOpensAt().plusDays(2));
		assertThat(slot.isTodayish()).isFalse();
	}

	@Test
	public void isTodayishTrue() {
		assertThat(slot.isTodayish()).isTrue();
	}
}
